/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.runtime.ws;

import org.jboss.soa.bpel.runtime.integration.ServerConfig;

import java.io.*;

class DeploymentBuilder
{
  private File war;
  private File webInf;
  private File wsdlDir;
  private DeploymentBuilder instance;

  private ServerConfig serverConfig;

  public DeploymentBuilder(ServerConfig serverConfig)
  {
    this.serverConfig = serverConfig;
  }

  public DeploymentBuilder setEndpoint(String endpointId)
  {
    File tmpDir = new File(serverConfig.getServerTempDir(), "riftsaw");

    tmpDir.mkdir();
    File fakeWebApp = new File(tmpDir, "riftsaw-"+endpointId+".war");
    File fakeWebInf = new File(fakeWebApp, "WEB-INF");
    fakeWebInf.mkdirs();

    // Workaround
    // See https://jira.jboss.org/jira/browse/JBWS-2718
    File fakeWSDLDir = new File(fakeWebInf, "wsdl");
    fakeWSDLDir.mkdirs();

    this.war = fakeWebApp;
    this.webInf = fakeWebInf;
    this.wsdlDir = fakeWSDLDir;

    return this;
  }

  public DeploymentBuilder setWSDL(File wsdl, File root)
  {
    copy(wsdl, new File(this.wsdlDir, wsdl.getName()));

    // any related artifact as well (brute force, I know)
    //File parent = wsdl.getParentFile();
    //assert parent.isDirectory();

    FileFilter filter = new FileFilter()
    {
      public boolean accept(File f)
      {
        return (true       		
        		/* Commented out, to ensure all appropriate artifacts are deployed
            f.getName().endsWith(".wsdl")
                || f.getName().endsWith(".xsd")
                || f.getName().endsWith(".xml")
                || f.getName().endsWith(".properties")
                || f.getName().endsWith(".jks")
                || f.getName().endsWith(".keystore")
                || f.getName().endsWith(".pem")
                || f.getName().endsWith(".chain")
                || f.isDirectory()
                */
        );
      }
    };

    for(File f : root.listFiles(filter))
    {
      if(f.equals(wsdl)) continue;
      copy(f, new File(this.wsdlDir, f.getName()));
      copy(f, new File(this.webInf, f.getName()));
      
      
/* Causes problems with testcase RIFTSAW-70 on jbossws-native. Ok
	on jbossws-cxf. So just copying all artifacts to both locations
	for now.

      if (f.getName().endsWith(".wsdl") ||
    		  f.getName().endsWith(".xsd")) {
    	  copy(f, new File(this.wsdlDir, f.getName()));
      } else {
    	  copy(f, new File(this.webInf, f.getName()));
     }
*/
    }
    return this;
  }
  
  public DeploymentBuilder setProvider(javax.xml.ws.Provider<?> provider) {
	  
	  // Check if jbossws-cxf.xml is present, and if so, updated the provider implementation class attribute
	  File f=new File(this.webInf, "jbossws-cxf.xml");
	  
	  if (f.exists()) {
		  FileInputStream fis=null;
		  FileOutputStream fos=null;
		  
		  try {
			  fis=new FileInputStream(f);
			  
			  byte[] b=new byte[fis.available()];
			  fis.read(b);
			  
			  String str=new String(b);
			  
			  fis.close();
			  fis = null;
			  
			  if (str.indexOf("@provider@") != -1) {
				  fos=new FileOutputStream(f);
				  
				  str = str.replaceAll("@provider@", provider.getClass().getName());
				  
				  fos.write(str.getBytes());
				  
				  fos.flush();
				  fos.close();
				  
				  fos = null;
			  } else {
				  // Report error
				  System.err.println("jbossws-cxf.xml file does not contain @provider@ field");
			  }
			  
		  } catch (IOException e) {
			  throw new RuntimeException("Failed to copy files", e);
		  } finally {
			  try {
				  if (fis != null) fis.close();
			  } catch (IOException e) {
			  }
			  try {
				  if (fos != null) fos.close();
			  } catch (IOException e) {
			  }
		  }
	  }
	  
	  return this;
  }

  public File build()
  {
    return this.war;
  }

  public static synchronized void copy(File src, File dest)
  {
    InputStream in = null;
    OutputStream out = null;

    try
    {
      if (src.isDirectory())
      {
        if (!dest.exists()) {
          dest.mkdir();
        }

        String[] children = src.list();
        for (int i=0; i<children.length; i++) {
          copy(new File(src, children[i]),
              new File(dest, children[i]));
        }
      } else {

        in = new FileInputStream(src);
        out = new FileOutputStream(dest);

        // Copy the bits from instream to outstream
        byte[] buf = new byte[1024];
        int len;
        while ((len = in.read(buf)) > 0) {
          out.write(buf, 0, len);
        }
        in.close();
        out.close();
      }

    }
    catch (IOException e)
    {
      throw new RuntimeException("Failed to copy files", e);
    }
    finally
    {
      try
      {
        if(in!=null) in.close();
      }
      catch (IOException e)
      {
        //
      }

      try
      {
        if(out!=null) out.close();
      }
      catch (IOException e)
      {
        //
      }
    }
  }
}
