/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.tests.samples;

import java.rmi.RemoteException;
import java.util.Calendar;
import java.util.GregorianCalendar;

import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import junit.framework.Test;

import org.jboss.soa.bpel.tests.RiftSawTest;
import org.jboss.soa.bpel.tests.RiftSawTestSetup;
import org.jboss.soa.bpel.tests.samples.travel.Cancelation;
import org.jboss.soa.bpel.tests.samples.travel.Detail;
import org.jboss.soa.bpel.tests.samples.travel.Flight;
import org.jboss.soa.bpel.tests.samples.travel.Hotel;
import org.jboss.soa.bpel.tests.samples.travel.Invoice;
import org.jboss.soa.bpel.tests.samples.travel.ItemSet;
import org.jboss.soa.bpel.tests.samples.travel.Order;
import org.jboss.soa.bpel.tests.samples.travel.Penalty;
import org.jboss.soa.bpel.tests.samples.travel.Query;
import org.jboss.soa.bpel.tests.samples.travel.RentalCar;
import org.jboss.soa.bpel.tests.samples.travel.TravelAgent;
import org.jboss.soa.bpel.tests.samples.travel.TravelService;

/**
 * @author Alejandro Guizar
 */
public class TutorialBPELTravelAgencyTestCase extends RiftSawTest {

	private TravelAgent travelAgent;

	private Flight flight = new Flight();
	private Hotel hotel = new Hotel();
	private RentalCar car = new RentalCar();
	private XMLGregorianCalendar tripDate;

	private static final String TEST_NAME = "Tutorial_bpel_travel_agency";

	public TutorialBPELTravelAgencyTestCase() {
		super(TEST_NAME);
	}

	public static Test suite() {
		return new RiftSawTestSetup(TutorialBPELTravelAgencyTestCase.class,
				TEST_NAME,
				"Tutorial_bpel_travel_agency.war,Tutorial_bpel_travel_agency-1.jar");
	}

	protected void setUp() throws Exception {
		TravelService tripService = new TravelService();
		travelAgent = tripService.getTravelAgentPort();

		flight.setAirline("AM");
		flight.setNumber(637);
		hotel.setName("Maria Isabel");
		car.setCompany("Alamo");

		GregorianCalendar now = new GregorianCalendar();
		now.add(Calendar.SECOND, 10);
		tripDate = DatatypeFactory.newInstance().newXMLGregorianCalendar(now);
	}

	public void testPurchaseTrip() throws RemoteException {
		ItemSet items = new ItemSet();
		items.setFlight(flight); // cost: 300
		items.setHotel(hotel); // cost: 100

		Order order = new Order();
		order.setDate(tripDate);
		order.setItems(items);

		Invoice invoice = travelAgent.purchaseTrip(order);
		assertEquals(300 + 100, invoice.getCost(), 0);
	}

	public void testGetTripDetails() throws RemoteException {
		ItemSet items = new ItemSet();
		items.setFlight(flight);
		items.setHotel(hotel);
		items.setRentalCar(car);

		Order order = new Order();
		order.setDate(tripDate);
		order.setItems(items);

		Invoice invoice = travelAgent.purchaseTrip(order);

		Query query = new Query();
		query.setLocator(invoice.getLocator());

		Detail detail = travelAgent.getTripDetail(query);
		items = detail.getItems();

		assertEquals(flight.getAirline(), items.getFlight().getAirline());
		assertEquals(flight.getNumber(), items.getFlight().getNumber());
		assertEquals(hotel.getName(), items.getHotel().getName());
		assertEquals(car.getCompany(), items.getRentalCar().getCompany());
		assertEquals(invoice.getCost(), detail.getCost(), 0);
	}

	public void testCancelTrip() throws Exception {
		ItemSet items = new ItemSet();
		items.setFlight(flight); // fee: 100
		items.setRentalCar(car); // fee: 5

		Order order = new Order();
		order.setDate(tripDate);
		order.setItems(items);

		Invoice invoice = travelAgent.purchaseTrip(order);

		Cancelation reference = new Cancelation();
		reference.setLocator(invoice.getLocator());

		Penalty penalty = travelAgent.cancelTrip(reference);
		assertEquals(100 + 5, penalty.getFee(), 0);
	}
}
