/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.server.ebws;

import javax.management.ObjectName;

import junit.framework.Test;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.listeners.message.MessageValidationException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.test.JBossTestCase;

/**
 * Test EBWS invocations.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */
public class EBWSUnitTest extends JBossTestCase
{
    /**
     * The name of the deployment archive.
     */
    private static final String ESB_ARCHIVE = "ebws-test.esb" ;
    
    /**
     * Construct the test case with the specified name.
     * @param name The name of the test case.
     */
    public EBWSUnitTest(final String name)
    {
        super(name) ;
    }
    
    /**
     * Test for normal request/response.
     * @throws Exception For any failures.
     */
    public void testNormalRequestResponse()
        throws Exception
    {
        runRequestResponseTest(null, null, EBWSUtil.NORMAL_SERVICE) ;
    }
    
    /**
     * Test for alternate request/response.
     * @throws Exception For any failures.
     */
    public void testAlternateRequestResponse()
        throws Exception
    {
        runRequestResponseTest(EBWSUtil.REQUEST_LOCATION, EBWSUtil.RESPONSE_LOCATION, EBWSUtil.ALTERNATE_SERVICE) ;
    }
    
    /**
     * Test for normal invalid request.
     * @throws Exception For any failures.
     */
    public void testNormalInvalidRequest()
        throws Exception
    {
        runInvalidRequestTest(null, EBWSUtil.NORMAL_SERVICE) ;
    }
    
    /**
     * Test for alternate invalid request.
     * @throws Exception For any failures.
     */
    public void testAlternateInvalidRequest()
        throws Exception
    {
        runInvalidRequestTest(EBWSUtil.REQUEST_LOCATION, EBWSUtil.ALTERNATE_SERVICE) ;
    }
    
    /**
     * Test for normal invalid response.
     * @throws Exception For any failures.
     */
    public void testNormalInvalidResponse()
        throws Exception
    {
        runInvalidResponseTest(null, EBWSUtil.NORMAL_SERVICE) ;
    }
    
    /**
     * Test for alternate invalid response.
     * @throws Exception For any failures.
     */
    public void testAlternateInvalidResponse()
        throws Exception
    {
        runInvalidResponseTest(EBWSUtil.REQUEST_LOCATION, EBWSUtil.ALTERNATE_SERVICE) ;
    }
    
    /**
     * Test for normal valid response through SOAP.
     * @throws Exception for any failures.
     */
    public void testNormalSoapRequest()
        throws Exception
    {
        runSoapRequestTest(EBWSUtil.NORMAL_SOAP_ENDPOINT) ;
    }
    
    /**
     * Test for alternate valid response through SOAP.
     * @throws Exception for any failures.
     */
    public void testAlternateSoapRequest()
        throws Exception
    {
        runSoapRequestTest(EBWSUtil.ALTERNATE_SOAP_ENDPOINT) ;
    }
    
    /**
     * Test for normal valid response through SOAP with addressing.
     * @throws Exception for any failures.
     */
    public void testNormalSoapAddressingRequest()
        throws Exception
    {
        runSoapAddressingRequestTest(EBWSUtil.NORMAL_SOAP_ADDRESSING_ENDPOINT) ;
    }
    
    private void runRequestResponseTest(final String requestLocation, final String responseLocation, final String name)
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        if (requestLocation == null)
        {
            message.getBody().add(EBWSUtil.VALID_REQUEST) ;
        }
        else
        {
            message.getBody().add(requestLocation, EBWSUtil.VALID_REQUEST) ;
        }
        final ServiceInvoker serviceInvoker = new ServiceInvoker("TestEBWS", name) ;
        
        final Message responseMessage = serviceInvoker.deliverSync(message, 30000) ;
        
        final Object response ;
        if (responseLocation == null)
        {
            response = responseMessage.getBody().get();
        }
        else
        {
            response = responseMessage.getBody().get(responseLocation) ;
        }
        
        assertEquals("Response Message", EBWSUtil.VALID_RESPONSE, response) ;
        final String[] logs = waitForMessages(2) ;
        assertNotNull("Log message", logs) ;
        assertEquals("Log count", 2, logs.length) ;
        assertTrue("Request log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_REQUEST, logs[0])) ;
        assertTrue("Response log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_RESPONSE, logs[1])) ;
    }
    
    private void runInvalidRequestTest(final String requestLocation, final String name)
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        if (requestLocation == null)
        {
            message.getBody().add(EBWSUtil.INVALID_REQUEST) ;
        }
        else
        {
            message.getBody().add(requestLocation, EBWSUtil.INVALID_REQUEST) ;
        }
        final ServiceInvoker serviceInvoker = new ServiceInvoker("TestEBWS", name) ;
        
        try
        {
            serviceInvoker.deliverSync(message, 30000) ;
            fail("Expected fault message") ;
        }
        catch (final FaultMessageException fme)
        {
            assertTrue("MessageValidationException cause, ", fme.getCause() instanceof MessageValidationException) ;
        }
    }
    
    private void runInvalidResponseTest(final String requestLocation, final String name)
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        if (requestLocation == null)
        {
            message.getBody().add(EBWSUtil.VALID_REQUEST) ;
        }
        else
        {
            message.getBody().add(requestLocation, EBWSUtil.VALID_REQUEST) ;
        }
        message.getBody().add(EBWSUtil.ERROR, "true") ;
        
        final ServiceInvoker serviceInvoker = new ServiceInvoker("TestEBWS", name) ;
        
        try
        {
            serviceInvoker.deliverSync(message, 30000) ;
            fail("Expected fault message") ;
        }
        catch (final FaultMessageException fme)
        {
            assertTrue("MessageValidationException cause, ", fme.getCause() instanceof MessageValidationException) ;
        }
        
        final String[] logs = waitForMessages(2) ;
        assertNotNull("Log message", logs) ;
        assertEquals("Log count", 2, logs.length) ;
        assertTrue("Request log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_REQUEST, logs[0])) ;
        assertTrue("Response log message", XMLHelper.compareXMLContent(EBWSUtil.INVALID_RESPONSE, logs[1])) ;
    }
    
    private void runSoapRequestTest(final String endpoint)
        throws Exception
    {
        final HttpClient httpClient = new HttpClient() ;
        final PostMethod postMethod = new PostMethod(endpoint) ;
        final StringRequestEntity request = new StringRequestEntity(EBWSUtil.SOAP_REQUEST) ;
        postMethod.setRequestEntity(request) ;
        
        httpClient.executeMethod(postMethod) ;
        final String response = postMethod.getResponseBodyAsString() ;
        System.out.println("Response is " + response) ;
        assertTrue("XML response", XMLHelper.compareXMLContent(EBWSUtil.SOAP_RESPONSE, response)) ;

        final String[] logs = waitForMessages(2) ;
        assertNotNull("Log message", logs) ;
        assertEquals("Log count", 2, logs.length) ;
        assertTrue("Request log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_REQUEST, logs[0])) ;
        assertTrue("Response log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_RESPONSE, logs[1])) ;
    }
    
    private void runSoapAddressingRequestTest(final String endpoint)
        throws Exception
    {
        final HttpClient httpClient = new HttpClient() ;
        final PostMethod postMethod = new PostMethod(endpoint) ;
        final StringRequestEntity request = new StringRequestEntity(EBWSUtil.SOAP_ADDRESSING_REQUEST) ;
        postMethod.setRequestEntity(request) ;
        
        httpClient.executeMethod(postMethod) ;
        final String response = postMethod.getResponseBodyAsString() ;
        System.out.println("Response is " + response) ;
        assertNotNull("XML response", response) ;
        assertTrue("XML response", response.contains("http://soa.jboss.org/TestEBWS/addressingOpResp")) ;

        final String[] logs = waitForMessages(3) ;
        assertNotNull("Log message", logs) ;
        assertEquals("Log count", 3, logs.length) ;
        assertTrue("Request log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_REQUEST, logs[0])) ;
        assertEquals("WS-A Message ID", "urn:esb:1234567", logs[1]) ;
        assertTrue("Response log message", XMLHelper.compareXMLContent(EBWSUtil.VALID_RESPONSE, logs[2])) ;
    }
    
    /**
     * Create the test suite.
     * @return The suite representing this test case.
     */
    public static Test suite()
        throws Exception
    {
        return getDeploySetup(EBWSUnitTest.class, ESB_ARCHIVE);
    }

    private String[] waitForMessages(final int count)
        throws Exception
    {
        return (String[]) getServer().invoke(new ObjectName(EBWSServerMBean.OBJECT_NAME), "waitForMessages", 
            new Integer[] {Integer.valueOf(count)}, new String[] { Integer.TYPE.getName() });
    }
}
