/*
 * RHQ Management Platform
 * Copyright (C) 2005-2008 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, * but WITHOUT ANY WARRANTY; without even the implied warranty of * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.jbosson.plugins.jbossesb;

import java.util.Set;

import org.mc4j.ems.connection.bean.attribute.EmsAttribute;
import org.rhq.core.domain.measurement.AvailabilityType;
import org.rhq.core.domain.measurement.MeasurementDataNumeric;
import org.rhq.core.domain.measurement.MeasurementReport;
import org.rhq.core.domain.measurement.MeasurementScheduleRequest;
import org.rhq.core.pluginapi.inventory.ResourceContext;
import org.rhq.plugins.jmx.MBeanResourceComponent;

import org.rhq.core.domain.configuration.Configuration;
import org.rhq.core.domain.configuration.PropertySimple;
import org.mc4j.ems.connection.bean.EmsBean;

/**
 * The ActionComponent finds ESB action data (messages, bytes, time).
 * 
 * @author Tom Cunningham
 */
public class ActionComponent extends MBeanResourceComponent<MBeanResourceComponent> {
    private ResourceContext<MBeanResourceComponent> context;

    private static final String SUCCESS_PROCESSED = "messages successfully processed count";
    private static final String FAILED_PROCESSED = "messages failed count";
    private static final String PROCESSING_TIME = "processing time";
    private static final String OVERALL_BYTES_PROCESSED = "processed bytes";
    private static final String OVERALL_BYTES_FAILED = "failed bytes";
    
    private static final String OVERALL_METRIC_NAME = "overallMessagesCount";
    private static final String SUCCESS_METRIC_NAME = "successProcessedCount";
    private static final String FAILED_METRIC_NAME = "failedProcessedCount";
  
    private static final String PROCESSINGTIME_METRIC_NAME = "processingTime";
    private static final String OVERALL_BYTES_METRIC_NAME = "overallBytesProcessed";
    private static final String BYTESPROCESSED_METRIC_NAME = "bytesSuccessful";
    private static final String BYTESFAILED_METRIC_NAME = "overallBytesFailed";

    private static final String OVERALL_MINUTE_METRIC_NAME = "overallMessagesCountMinute";
    private static final String SUCCESS_MINUTE_METRIC_NAME = "successProcessedCountMinute";
    private static final String FAILED_MINUTE_METRIC_NAME = "failedProcessedCountMinute";
    
    /**
     * Is this service alive?
     *
     * @return true if the service is running
     */
    @Override
	public AvailabilityType getAvailability() {
        try {
        	EmsBean emsbean = getEmsBean();
        	if (emsbean.isRegistered()) {
            	String actionName = this.context.getResourceKey();
            	actionName = actionName.trim();
        		EmsAttribute attribute = this.bean.getAttribute(actionName + " " + SUCCESS_PROCESSED);
        		if (attribute != null) {
        			return AvailabilityType.UP;
        		}
        		return AvailabilityType.DOWN;
        	} else {
        		return AvailabilityType.DOWN;
        	}
        } catch (NullPointerException npe) {
            if (resourceContext != null) {
                log.warn("Could not determine availability of unknown ems bean for ["
                    + resourceContext.getResourceType() + ":" + resourceContext.getResourceKey() + "]");
            }

            return AvailabilityType.DOWN;
        }
    }
    
    @Override
    public void start(ResourceContext<MBeanResourceComponent> context) {
       	super.start(context); 
       	this.context = context;
        this.bean = context.getParentResourceComponent().getEmsBean();
    }

    @Override
    public void getValues(MeasurementReport report, Set<MeasurementScheduleRequest> requests) {
    	Configuration pluginConfig = this.context.getPluginConfiguration();
    	pluginConfig.put(new PropertySimple("type", "action"));
    	//String actionName = pluginConfig.getActionName();
	
    	String actionName = this.context.getResourceKey();
    	actionName = actionName.trim();
        //Object entityStatistics = operation.invoke(context.getResourceKey());

    	int counter = 0;
    	int success = 0;
    	int failed = 0;
    	
    	long counterBytesFailed = 0;
    	long counterBytesProcessed = 0;
    	for (MeasurementScheduleRequest request : requests) {
        	String metricName = request.getName();
        	if (metricName.equals (SUCCESS_METRIC_NAME)) {
        		EmsAttribute attribute = this.bean.getAttribute(actionName + " " + SUCCESS_PROCESSED);
        		Integer processed = new Integer(0);
        		if (attribute != null) {
        			processed = (Integer) attribute.refresh();
        		}
        		success = processed.intValue();
        		counter += processed.intValue();
        		report.addData(new MeasurementDataNumeric(request, new Double(processed.doubleValue())));
        	} else if (metricName.equals(FAILED_METRIC_NAME)) {
        		EmsAttribute attribute = this.bean.getAttribute(actionName + " " + FAILED_PROCESSED);
        	    Integer processed = new Integer(0);
        	    if (attribute != null) {
        	    	processed = (Integer) attribute.refresh();
        	    }
        	    failed = processed.intValue();
        	    counter += processed.intValue();
        		report.addData(new MeasurementDataNumeric(request, new Double(processed.doubleValue())));
        	} else if (metricName.equals(PROCESSINGTIME_METRIC_NAME)) {
        		EmsAttribute attribute = this.bean.getAttribute(actionName + " " + PROCESSING_TIME);
        		Double processed = (Double) attribute.refresh();
        		if (processed != null) {
            		processed = new Double(processed.doubleValue() / 1000000);
        			report.addData(new MeasurementDataNumeric(request, processed));
        		} else {
        			report.addData(new MeasurementDataNumeric(request, new Double(0)));
        		}
        	} else if (metricName.equals(BYTESPROCESSED_METRIC_NAME)) {
        		EmsAttribute attribute = this.bean.getAttribute(actionName + " " + OVERALL_BYTES_PROCESSED);
        		Long bytesProcessed = new Long(0);
        		if (attribute != null) {
        			bytesProcessed = (Long) attribute.refresh();
        			counterBytesProcessed = bytesProcessed.longValue();
        		}
        		report.addData(new MeasurementDataNumeric(request, new Double(bytesProcessed.doubleValue())));
        	} else if (metricName.equals(BYTESFAILED_METRIC_NAME)) {
        		EmsAttribute attribute = this.bean.getAttribute(actionName + " " + OVERALL_BYTES_FAILED);
        		Long bytesFailed = new Long(0);
        		if (attribute != null) {
        			bytesFailed = (Long) attribute.refresh();
        			counterBytesFailed += bytesFailed.longValue();
        		}
        		report.addData(new MeasurementDataNumeric(request, new Double(bytesFailed.doubleValue())));        		
        	}
        }
    	for (MeasurementScheduleRequest request : requests) {
        	String metricName = request.getName();
        	if (metricName.equals (OVERALL_METRIC_NAME)) {
        		report.addData(new MeasurementDataNumeric(request, new Double(counter)));        		
        	} else if (metricName.equals(OVERALL_MINUTE_METRIC_NAME)) {
        		report.addData(new MeasurementDataNumeric(request, new Double(counter)));        		
        	} else if (metricName.equals(SUCCESS_MINUTE_METRIC_NAME)) {
        		report.addData(new MeasurementDataNumeric(request, new Double(success)));        		        		
        	} else if (metricName.equals(FAILED_MINUTE_METRIC_NAME)) {
        		report.addData(new MeasurementDataNumeric(request, new Double(failed)));        		        		
        	} else if (metricName.equals(OVERALL_BYTES_METRIC_NAME)) {
        		report.addData(new MeasurementDataNumeric(request, new Double(counterBytesProcessed + counterBytesFailed)));
        	}
    	}
    }
}
