/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2010
 */
package org.jboss.soa.esb.actions.soap.proxy;

import static org.junit.Assert.assertEquals;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.util.Properties;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.actions.soap.proxy.SOAPProxyWsdlLoader.DefaultSOAPProxyWsdlLoader;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.jboss.soa.esb.http.configurators.Connection;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.junit.Test;

/**
 * SOAPProxy unit test.
 * 
 * @author dward at jboss.org
 */
public class SOAPProxyUnitTest
{
	
	@Test
	public void test_fixRelative()
	{
		String[][] relativeHttpResources = new String[][] {
				new String[] {"http://foo.com/services/MyService.wsdl", "/schemas/MySchema.xsd", "http://foo.com/schemas/MySchema.xsd"},
				new String[] {"http://foo.com/services/MyService.wsdl", "../schemas/MySchema.xsd", "http://foo.com/schemas/MySchema.xsd"},
				new String[] {"http://foo.com/services/MyService.wsdl", "schemas/MySchema.xsd", "http://foo.com/services/schemas/MySchema.xsd"},
				new String[] {"http://foo.com/services/MyService.wsdl", "/MySchema.xsd", "http://foo.com/MySchema.xsd"},
				new String[] {"http://foo.com/services/old/MyService.wsdl", "../new/MySchema.xsd", "http://foo.com/services/new/MySchema.xsd"},
				new String[] {"http://foo.com/services/MyService.wsdl", "/test/MySchema.xsd", "http://foo.com/test/MySchema.xsd"},
				new String[] {"http://foo.com/services/old/MyService.wsdl", "../../test/MySchema.xsd", "http://foo.com/test/MySchema.xsd"},
				new String[] {"http://foo.com/services/MyService.wsdl", "./MySchema.xsd", "http://foo.com/services/MySchema.xsd"},
				new String[] {"http://foo.com/services/old/MyService.wsdl", "./../test/MySchema.xsd", "http://foo.com/services/test/MySchema.xsd"},
				new String[] {"http://foo.com/services/old/MyService.wsdl", ".././test/MySchema.xsd", "http://foo.com/services/test/MySchema.xsd"},
				new String[] {"http://foo.com/services/old/MyService.wsdl", ".././../test/MySchema.xsd", "http://foo.com/test/MySchema.xsd"},
				new String[] {"http://foo.com/services/old/MyService.wsdl", ".././../test/./MySchema.xsd", "http://foo.com/test/MySchema.xsd"},
		};
		SOAPProxyWsdlLoader loader = new DefaultSOAPProxyWsdlLoader(new ConfigTree("config"), null, true);
		for (String[] r : relativeHttpResources)
		{
			String actual = loader.fixRelative(r[0], r[1]);
			String expected = r[2];
			assertEquals(expected, actual);
		}
	}
	
	@Test
	public void test_maxThreadsPropagation() throws Exception
	{
		final String expected = "100";
		ConfigTree parent_config = new ConfigTree("parent");
		parent_config.setAttribute(ListenerTagNames.MAX_THREADS_TAG, expected);
		ConfigTree child_config = new ConfigTree("child", parent_config);
		child_config.setAttribute("endpointUrl", "http://localhost"); // required attribute
		HttpSOAPProxyTransport transport = new HttpSOAPProxyTransport(child_config, null, null);
		Properties httpClientProps = transport.getHttpRouter().getHttpClientProps();
		final String actual_mtc = httpClientProps.getProperty(Connection.MAX_TOTAL_CONNECTIONS);
		assertEquals(expected, actual_mtc); 
		final String actual_mcph = httpClientProps.getProperty(Connection.MAX_CONNECTIONS_PER_HOST);
		assertEquals(expected, actual_mcph);
	}
	
	@Test
	public void test_maxThreadsPropertyOverrides() throws Exception
	{
		final String expected_mtc = "100";
		final String expected_mcph = "50";
		ConfigTree parent_config = new ConfigTree("parent");
		parent_config.setAttribute(ListenerTagNames.MAX_THREADS_TAG, "1");
		ConfigTree child_config = new ConfigTree("child", parent_config);
		child_config.setAttribute("endpointUrl", "http://localhost"); // required attribute
		ConfigTree mtc_config = new ConfigTree(HttpClientFactory.HTTP_CLIENT_PROPERTY, child_config);
		mtc_config.setAttribute("name", Connection.MAX_TOTAL_CONNECTIONS);
		mtc_config.setAttribute("value", expected_mtc);
		ConfigTree mcph_config = new ConfigTree(HttpClientFactory.HTTP_CLIENT_PROPERTY, child_config);
		mcph_config.setAttribute("name", Connection.MAX_CONNECTIONS_PER_HOST);
		mcph_config.setAttribute("value", expected_mcph);
		HttpSOAPProxyTransport transport = new HttpSOAPProxyTransport(child_config, null, null);
		Properties httpClientProps = transport.getHttpRouter().getHttpClientProps();
		final String actual_mtc = httpClientProps.getProperty(Connection.MAX_TOTAL_CONNECTIONS);
		assertEquals(expected_mtc, actual_mtc); 
		final String actual_mcph = httpClientProps.getProperty(Connection.MAX_CONNECTIONS_PER_HOST);
		assertEquals(expected_mcph, actual_mcph);
	}
	
	@Test
	public void test_maxThreadsFileOverrides() throws Exception
	{
		final String expected_mtc = "100";
		final String expected_mcph = "50";
		ConfigTree parent_config = new ConfigTree("parent");
		parent_config.setAttribute(ListenerTagNames.MAX_THREADS_TAG, "1");
		ConfigTree child_config = new ConfigTree("child", parent_config);
		child_config.setAttribute("endpointUrl", "http://localhost"); // required attribute
		File file = null;
		OutputStream os = null;
		try
		{
			file = File.createTempFile(HttpClientFactory.HTTP_CLIENT_PROPERTY, ".properties");
			os = new BufferedOutputStream(new FileOutputStream(file));
			Properties fileProperties = new Properties();
			fileProperties.setProperty(Connection.MAX_TOTAL_CONNECTIONS, expected_mtc);
			fileProperties.setProperty(Connection.MAX_CONNECTIONS_PER_HOST, expected_mcph);
			fileProperties.store(os, getClass().getName());
			os.flush(); os.close(); os = null;
			child_config.setAttribute("file", file.getAbsolutePath());	
			HttpSOAPProxyTransport transport = new HttpSOAPProxyTransport(child_config, null, null);
			Properties httpClientProps = transport.getHttpRouter().getHttpClientProps();
			final String actual_mtc = httpClientProps.getProperty(Connection.MAX_TOTAL_CONNECTIONS);
			assertEquals(expected_mtc, actual_mtc); 
			final String actual_mcph = httpClientProps.getProperty(Connection.MAX_CONNECTIONS_PER_HOST);
			assertEquals(expected_mcph, actual_mcph);
		}
		finally
		{
			try { if (os != null) os.close(); } catch (Throwable t) {}
			try { if (file != null) file.delete(); } catch (Throwable t) {}
		}
	}
	
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(SOAPProxyUnitTest.class);
    }

}
