/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

package org.jboss.soa.esb.actions.soap.adapter;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.wsdl.Definition;
import javax.wsdl.WSDLException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.JBossDeployerUtil;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.soa.esb.actions.soap.JBossWSDLLocatorImpl;
import org.jboss.soa.esb.actions.soap.JBossWSDLReader;
import org.jboss.soa.esb.actions.soap.adapter.wsnative.NativeFactory;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.ws.core.utils.JBossWSEntityResolver;
import org.jboss.wsf.spi.deployment.Deployment;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.xml.sax.EntityResolver;

/**
 * Factory for providing a common abstraction for all native/CXF functions.
 *  
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public abstract class JBossWSFactory
{
    /**
     * The logger for the soap processor factory implementations.
     */
    private static final Logger LOGGER = Logger.getLogger(JBossWSFactory.class) ;
    /**
     * The internal URI that should be returned as the WSDL location if a subsequent request
     * should be make through the getWsdl method.
     */
    public static final String ESB_INTERNAL_URI = "esbinternal" ;
    
    /**
     * The implementation factory.
     */
    private static final JBossWSFactory FACTORY ;
    
    /**
     * Get the factory singleton.
     * @return The factory singleton.
     */
    public static JBossWSFactory getFactory()
    {
        return FACTORY ;
    }
    
    /**
     * Get the SOAP Processor factory for the specific implementation.
     * @return The SOAP Processor factory.
     */
    public abstract SOAPProcessorFactory getSOAPProcessorFactory() ;
    
    /**
     * Get the URI representing the WSDL location for the specified endpoint
     * @param endpoint The specific endpoint.
     * @return The WSDL location or null if unknown.
     */
    public abstract String getWsdlLocation(final Endpoint endpoint, final String endpointName) ;
    
    /**
     * Get the WSDL for the specified endpoint.
     * @param uri The URI representing the endpoint.
     * @return The WSDL or null if unknown.
     * @throws IOException for errors creating the stream
     */
    public abstract InputStream getWsdl(final String uri)
        throws IOException ;
    
    /**
     * Get the definition for the WSDL at the specific location.
     * @param wsdlURL The WSDL URL.
     * @return The definition.
     * @throws WSDLException For errors parsing the WSDL.
     */
    public abstract Definition readWSDL(URL wsdlURL)
        throws WSDLException ;
    
    /**
     * Get the context ClassLoader for the specified endpoint
     * @param endpoint The specific endpoint.
     * @return The context ClassLoader.
     */
    public ClassLoader getClassLoader(final Endpoint endpoint)
    {
        if (JBossDeployerUtil.isWebMetaDataPresent())
        {
            final Deployment deployment = endpoint.getService().getDeployment();
            final JBossWebMetaData attachment = deployment.getAttachment(JBossWebMetaData.class);
            
            if (attachment != null)
            {
                return attachment.getENCLoader() ;
            }
        }
        return null ;
    }
    
    static
    {
        JBossWSFactory factory = null ;

        if (JBossDeployerUtil.getWSImpl().equals(JBossDeployerUtil.WSIMPL_CXF))
        {
            try
            {
                // Load using reflection as it will not be present on 1.5 compilations
                final Class<?> factoryClass = ClassUtil.forName("org.jboss.soa.esb.actions.soap.adapter.cxf.CXFFactory", JBossWSFactory.class) ;
                factory = JBossWSFactory.class.cast(factoryClass.newInstance()) ;
            }
            catch (final Throwable th)
            {
                if (LOGGER.isDebugEnabled())
                {
                    LOGGER.debug("Failed to load CXF factory", th) ;
                }
            }
        }
        if (factory == null)
        {
            factory = new NativeFactory() ;
        }
        FACTORY = factory ;
    }
}
