/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.smooks;

import junit.framework.TestCase;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.smooks.message.Person1;
import org.jboss.soa.esb.smooks.message.Person2;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.milyn.container.plugin.SourceResult;
import org.milyn.payload.ByteSource;
import org.milyn.payload.StringResult;

import java.io.ByteArrayInputStream;
import java.io.StringReader;

/**
 * More SmooksAction unit tests.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SourceResultUnitTest extends TestCase {
    
    private ESBConfigUtil esbConfig;
    private byte[] personXmlBytes = StreamUtils.readStream(SourceResultUnitTest.class.getResourceAsStream("message/person.xml"));

    protected void setUp() throws Exception {
        esbConfig = new ESBConfigUtil(getClass().getResourceAsStream("source-result-esb.xml"));
        esbConfig.setESBProperties(getClass().getResourceAsStream("jbossesb-properties.xml"));
        esbConfig.installRegistry();
        esbConfig.startController();
    }

    protected void tearDown() throws Exception {
        esbConfig.stopController();
        esbConfig.uninstallRegistry();
        esbConfig.resetESBProperties();
    }

    public void test_String2String() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "String");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(new String(personXmlBytes));
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom Fennelly\" is=\"21\">", message.getBody().get());
    }

    public void test_bytes2String() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "String");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(personXmlBytes);
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom Fennelly\" is=\"21\">", message.getBody().get());
    }

    public void test_InputStream2String() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "String");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(new ByteArrayInputStream(personXmlBytes));
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom Fennelly\" is=\"21\">", message.getBody().get());
    }

    public void test_Reader2String() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "String");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(new StringReader(new String(personXmlBytes)));
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom Fennelly\" is=\"21\">", message.getBody().get());
    }

    public void test_String2Bytes() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "Bytes");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(new String(personXmlBytes));
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom Fennelly\" is=\"21\">", new String((byte[]) message.getBody().get()));
    }

    public void test_Object2String() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "Person");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(new Person1("Tom", 300));
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom\" is=\"300\">", message.getBody().get());
    }

    public void test_Object2Object() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "Person2Person");
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(new Person1("Tom", 300));
        message = invoker.deliverSync(message, 30000);

        Person2 person2 = (Person2) message.getBody().get();
        assertEquals("Tom", person2.getName());
        assertEquals(300, person2.getAge());
    }

    public void test_SourceResult() throws MessageDeliverException, RegistryException, FaultMessageException {
        ServiceInvoker invoker = new ServiceInvoker("Transform", "String");
        Message message = MessageFactory.getInstance().getMessage();
        SourceResult sourceResult = new SourceResult();
        StringResult result = new StringResult();

        sourceResult.setSource(new ByteSource(personXmlBytes));
        sourceResult.setResult(result);

        message.getBody().add(sourceResult);
        message = invoker.deliverSync(message, 30000);

        assertEquals("<someone called=\"Tom Fennelly\" is=\"21\">", result.getResult());
    }
}
