/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.smooks;

import static org.junit.Assert.*;

import java.util.Hashtable;
import java.util.Map;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Before;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/**
 * Unit test for {@link org.jboss.soa.esb.smooks.SmooksAction}
 * 
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>			
 *
 */
public class SmooksActionUnitTest
{
	private String expectedString = "<test></test>";
	private ConfigTree config;
	
	@Test
	public void process() throws ConfigurationException, ActionProcessingException
	{
		SmooksAction action = new SmooksAction( config );
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( expectedString );
		
		Message process = action.process( message );
		
		String actualString = (String)process.getBody().get();
		assertEquals( expectedString, actualString );
	}
	
	@Test
	public void processWithPayloadLocation() throws ConfigurationException, ActionProcessingException
	{
		config.setAttribute( MessagePayloadProxy.GET_PAYLOAD_LOCATION, "input" );
		SmooksAction action = new SmooksAction( config );
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( "input", expectedString );
		
		Message process = action.process( message );
		
		String actualString = (String)process.getBody().get();
		assertEquals( expectedString, actualString );
	}
	
	@Test
	public void processWithOutLocation() throws ConfigurationException, ActionProcessingException
	{
		config.setAttribute( MessagePayloadProxy.SET_PAYLOAD_LOCATION, "output" );
		SmooksAction action = new SmooksAction( config );
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( expectedString );
		
		Message process = action.process( message );
		
		String actualString = (String)process.getBody().get( "output" );
		assertEquals( expectedString, actualString );
	}
	
	@Test
	public void getExecutionContextAttributes() throws ConfigurationException, ActionProcessingException
	{
		SmooksAction action = new SmooksAction( config );
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( expectedString );
		
		Message process = action.process( message );
		
		Object object = process.getBody().get( SmooksAction.EXECUTION_CONTEXT_ATTR_MAP_KEY );
		assertTrue( object instanceof Map );
	}
	
	@Test
	@SuppressWarnings("unchecked")
	public void getExecutionContextAttributesNonSerializable() throws ConfigurationException, ActionProcessingException
	{
		Map map = new Hashtable();
		map.put( "test1", "testing" );
		map.put( "test2", new NonSerializableClass() );

        ConfigTree configTree = createConfigTree();
        configTree.setAttribute("mappedContextObjects", "test1,test2");
        SmooksAction action = new SmooksAction( configTree );
		
		Map serializableObjects = action.getSerializableObjectsMap( map );
		
		assertTrue( serializableObjects.containsKey( "test1" ));
		assertTrue( !serializableObjects.containsKey( "test2" ));
	}

    @Test
    public void ConstructorResultType() throws ConfigurationException, ActionProcessingException
    {
        config.setAttribute( "resultType", "STRING" );
        new SmooksAction( config );
        config.setAttribute( "resultType", "BYTES" );
        new SmooksAction( config );
        config.setAttribute( "resultType", "JAVA" );
        new SmooksAction( config );
        config.setAttribute( "resultType", "NORESULT" );
        new SmooksAction( config );

        config.setAttribute( "resultType", "BLAHHH" );
        try {
            new SmooksAction( config );
            fail("Expected ConfigurationException");
        } catch (ConfigurationException e) {
            assertEquals("Invalid 'resultType' config value 'BLAHHH'.  Valid values are: [STRING, BYTES, JAVA, NORESULT]", e.getMessage());
        }
    }

	@Before
	public void setup()
	{
		config = createConfigTree();
		
	}
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( SmooksActionUnitTest.class );
	}
	
	private ConfigTree createConfigTree()
	{
		ConfigTree config = new ConfigTree( "SmooksActionTest" );
		config.setAttribute( "smooksConfig", "/org/jboss/soa/esb/smooks/smooks-config.xml");
		return config;
	}
	
	private static class NonSerializableClass 
	{
		
	}

}
