/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.CONTINUE;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.DISPOSE;
import static org.junit.Assert.assertEquals;

import java.math.BigDecimal;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.internal.soa.esb.services.routing.cbr.Order;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for {@link BusinessRulesProcessor}
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class BusinessRulesProcessorUnitTest
{
	private Order order;
	private ArrayList<String> messagePathList;

	@Test
	public void processDiscount() throws ObjectMappingException, ConfigurationException, RegistryException, MessageRouterException, ActionProcessingException
	{
		Message message = createMessageWithOrder( order );
        ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBPricingRulesStateful.drl" ).messagePaths(messagePathList).build();
		BusinessRulesProcessor processor = new BusinessRulesProcessor( configTree );

		processor.process( message );

        assertEquals( 20.0, order.getDiscount(), 0);
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));
	}

	@Test
	public void processDiscountStateful() throws ObjectMappingException, ConfigurationException, RegistryException, MessageRouterException, ActionProcessingException
	{
		Message message = createMessageWithOrder( order );
        ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBPricingRulesStateful.drl" ).stateful( true ).messagePaths(messagePathList).build();
		BusinessRulesProcessor processor = new BusinessRulesProcessor( configTree );

		// process message
		processor.process( message );
        assertEquals( 20.0, order.getDiscount(), 0);
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));

        //	now dispose after this call
        message.getProperties().setProperty( DISPOSE.getName(), Boolean.TRUE );
        message.getProperties().setProperty( CONTINUE.getName(), Boolean.TRUE );
		processor.process( message );

		//	counter is inserted into the working memory by the first rules execution
		//	from the above process call.
        Counter counter = (Counter) message.getBody().get("Counter");
        assertEquals( 2 , counter.getCounter() );
	}

    @Test
    public void processStatefulDispose() throws Exception
    {
        Message message = createMessageWithOrder( order );
        ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBStatefulDispose.drl" ).stateful( true ).messagePaths(messagePathList).build();
        BusinessRulesProcessor processor = new BusinessRulesProcessor( configTree );

        // process message
        processor.process( message );
        assertEquals( 20.0, order.getDiscount(), 0);
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));
        Counter counter = (Counter) message.getBody().get("Counter");
        assertEquals( 1 , counter.getCounter() );

        //  do not dispose after this call
        message.getBody().remove("Order");
        message.getProperties().setProperty( DISPOSE.getName(), Boolean.FALSE );
        message.getProperties().setProperty( CONTINUE.getName(), Boolean.TRUE );
        message.getBody().add("Counter", new Counter(counter.getCounter()));
        processor.process( message );

        counter = (Counter) message.getBody().get("Counter");
        assertEquals( 2 , counter.getCounter() );
        
        //  now dispose after this call
        message.getProperties().setProperty( DISPOSE.getName(), Boolean.TRUE );
        message.getProperties().setProperty( CONTINUE.getName(), Boolean.TRUE );
        message.getBody().add("Counter", new Counter(counter.getCounter()));
        processor.process( message );

        // If this assert fails then the second invocation disposed of the state when it shouldn't have
        // and this one did not fire any rules.
        counter = (Counter) message.getBody().get("Counter");
        assertEquals( 3 , counter.getCounter() );
        
        final Order order = new Order();
        order.setQuantity(20);
        order.setUnitPrice( new BigDecimal("20.0") );
        message = createMessageWithOrder( order );
        
        message.getProperties().setProperty( DISPOSE.getName(), Boolean.TRUE );
        message.getProperties().setProperty( CONTINUE.getName(), Boolean.TRUE );
        processor.process( message );

        assertEquals( 20.0, order.getDiscount(), 0);
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));
        counter = (Counter) message.getBody().get("Counter");
        assertEquals( 1 , counter.getCounter() );
    }

    @Test
    public void processStatefulContinuation() throws Exception
    {
        Message message = createMessageWithOrder( order );
        ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBStatefulDispose.drl" ).stateful( true ).messagePaths(messagePathList).build();
        BusinessRulesProcessor processor = new BusinessRulesProcessor( configTree );

        // process message
        processor.process( message );
        assertEquals( 20.0, order.getDiscount(), 0);
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));
        Counter counter = (Counter) message.getBody().get("Counter");
        assertEquals( 1 , counter.getCounter() );

        //  do not dispose after this call
        message.getBody().remove("Order");
        message.getProperties().setProperty( DISPOSE.getName(), Boolean.FALSE );
        message.getProperties().setProperty( CONTINUE.getName(), Boolean.TRUE );
        message.getBody().add("Counter", new Counter(counter.getCounter()));
        processor.process( message );

        counter = (Counter) message.getBody().get("Counter");
        assertEquals( 2 , counter.getCounter() );
        
        //  Do not specify continue, it should dispose current session before this call
        final Order order = new Order();
        order.setQuantity(20);
        order.setUnitPrice( new BigDecimal("20.0") );
        message = createMessageWithOrder( order );
        
        processor.process( message );

        assertEquals( 20.0, order.getDiscount(), 0);
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));
        counter = (Counter) message.getBody().get("Counter");
        assertEquals( 1 , counter.getCounter() );
    }

	@Test ( expected = ActionProcessingException.class )
	public void shouldThrowIfDisposePropertyIsNotSet() throws ObjectMappingException, ConfigurationException, RegistryException, MessageRouterException, ActionProcessingException
	{
		Message message = createMessageWithOrder( order );
        ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBPricingRulesStateful.drl" ).messagePaths(messagePathList).stateful( true ).build();
		BusinessRulesProcessor processor = new BusinessRulesProcessor( configTree );

		// process message
		processor.process( message );
        assertEquals( 20.0, order.getDiscount(), 0 );
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));

        //	We don't set 'dispose' which should cause an exception to be thrown...
        message.getProperties().setProperty( CONTINUE.getName(), Boolean.TRUE );
		processor.process( message );
	}

	@Test
    public void processDiscountWithEntryPoint() throws ObjectMappingException, ConfigurationException, RegistryException, MessageRouterException, ActionProcessingException
    {
		Order order = new Order();
        order.setQuantity(20);
        order.setUnitPrice( new BigDecimal("20.0") );

        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add("Order", order);

        final ArrayList<String> defaultFacts = new ArrayList<String>();
        defaultFacts.add("body.Counter");

        final ArrayList<String> orderEntryPoints = new ArrayList<String>();
        orderEntryPoints.add("body.Order");

        Map<String, List<String>> entryPointFacts = new HashMap<String, List<String>>();
        entryPointFacts.put("OrderEntryPoint", orderEntryPoints);
        CBRConfigTreeBuilder builder = new CBRConfigTreeBuilder(true).ruleFile("PricingRulesStatefulEntryPoint.drl");
        builder.messagePaths(defaultFacts).entryPoints(entryPointFacts);
        ConfigTree config = builder.stateful(true).build();
        BusinessRulesProcessor processor = new BusinessRulesProcessor(config);

        processor.process( message );

        assertEquals( 20.0, order.getDiscount(), 0 );
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));
    }

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(BusinessRulesProcessorUnitTest.class);
	}

	@Before
	public void setup()
	{
		order = new Order();
        order.setQuantity(20);
        order.setUnitPrice( new BigDecimal("20.0") );

        messagePathList = new ArrayList<String>();
        messagePathList.add("body.Order");
        messagePathList.add("body.Counter");
	}

	@BeforeClass
	public static void setupMockRegistry() throws URISyntaxException
	{
        MockRegistry.install();
	}

	private Message createMessageWithOrder( final Order order )
	{
		Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		message.getBody().add("Order", order);
		return message;
	}

}
