/*
* JBoss, Home of Professional Open Source
* Copyright 2008, Red Hat Middleware LLC, and individual contributors
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.services.routing.cbr;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.xml.xpath.XPathExpressionException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.services.rules.util.RulesContext;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for {@link DslHelper} 
 * <p/> 
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class DslHelperUnitTest
{
	private Logger log = Logger.getLogger( DslHelperUnitTest.class );
	
	// message with xml content in body
	private static Message msg;
	
	// message with namespace context in body 
	private static Message nsmsg;
	private static HashMap<String, String> namespaces;

	@Test
	public void xmlContentExists() throws XPathExpressionException, UnsupportedEncodingException
	{
		assertTrue( DslHelper.xmlContentExists( msg, "//*[@productId = 299]" ) );
		assertTrue( DslHelper.xmlContentExists( msg, "//*[@productId < 1000]" ) );
		assertFalse( DslHelper.xmlContentExists( msg, "//*[@productId = 2299]" ) );
	}
	
	@Test
	public void xmlContentExistsWithNamespaces() throws XPathExpressionException, UnsupportedEncodingException
	{
		assertTrue( DslHelper.xmlContentExists( nsmsg, "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product", namespaces ) );
		assertTrue( DslHelper.xmlContentExists( nsmsg, "//p:Product", namespaces ) );
		
        HashMap<String, String> oneNamspace = new HashMap<String,String>();
        oneNamspace.put( "ord", "http://org.jboss.soa.esb/order" );
		assertTrue( DslHelper.xmlContentExists( nsmsg, "/ord:Order", oneNamspace ) );
	}
	
	@Test
	public void xmlContentMatches() throws XPathExpressionException, UnsupportedEncodingException
	{
		assertTrue( DslHelper.xmlContentMatches( msg, "//*[@productId = 299]" ) );
		assertTrue( DslHelper.xmlContentMatches( msg, "/Order/OrderLines/OrderLine/Product" ) );
		assertFalse( DslHelper.xmlContentMatches( msg, "//*[@productId > 1299]" ) );
	}
	
	@Test
	public void xmlContentMatchesWithNamespaces() throws XPathExpressionException, UnsupportedEncodingException
	{
		assertTrue( DslHelper.xmlContentMatches( nsmsg, "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product", namespaces ) );
	}
	
	@Test
	public void selectAsBoolean() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.selectAsBoolean( msg, "/Order/OrderLines/OrderLine/Product/@productId = 364" ) );
		assertTrue( DslHelper.selectAsBoolean( msg, "//*[@productId = 299]" ) );
		assertTrue( DslHelper.selectAsBoolean( msg, "//*[@productId < 1299]" ) );
		assertFalse( DslHelper.selectAsBoolean( msg, "/Order/OrderLines/OrderLine/Product/@productId = 33" ) );
	}
	
	@Test
	public void selectAsBooleanWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.selectAsBoolean( nsmsg, "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product/@productId = 364", namespaces ) );
	}
	
	@Test
	public void selectAsNumber() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( 364, DslHelper.selectAsNumber( msg, "//Product/@productId" ).intValue() );
	}
	
	@Test
	public void selectAsNumberWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( 364, DslHelper.selectAsNumber( msg, "//Product/@productId", namespaces ).intValue() );
	}
	
	@Test
	public void selectAsString() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( "364", DslHelper.selectAsString( msg, "//Product/@productId" ) );
	}
	
	@Test
	public void selectAsStringWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( "364", DslHelper.selectAsString( nsmsg, "//p:Product/@productId", namespaces ) );
	}
	
	@Test
	public void selectAsNode() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( "productId",  DslHelper.selectAsNode( msg, "/Order/OrderLines/OrderLine/Product/@productId").getNodeName() );
	}
	
	@Test
	public void selectAsNodeWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( "productId",  DslHelper.selectAsNode( nsmsg, "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product/@productId", namespaces ).getNodeName() );
	}
	
	@Test
	public void selectAsNodeList() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( 42, DslHelper.selectAsNodeList( msg, "//Product/@productId" ).getLength() );
	}
	
	@Test
	public void selectAsNodeListWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertEquals ( 2, DslHelper.selectAsNodeList( nsmsg, "//p:Product/@productId", namespaces ).getLength() );
	}
	
	@Test
	public void xmlContentEquals() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.xmlContentEquals( msg, "/Order/OrderLines/OrderLine/Product/@productId", "364" ) );
		assertFalse( DslHelper.xmlContentEquals( msg, "/Order/OrderLines/OrderLine/Product/@productId", "1" ) );
	}
	
	@Test
	public void xmlContentEqualsWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.xmlContentEquals( nsmsg, "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product/@productId", "364", namespaces ) );
	}
	
	@Test
	public void xmlContentGreaterThan() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.xmlContentGreaterThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "363" ) );
		assertFalse( DslHelper.xmlContentGreaterThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "365" ) );
		assertFalse( DslHelper.xmlContentGreaterThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "364" ) );
	}
	
	@Test
	public void xmlContentGreaterThanWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.xmlContentGreaterThan( nsmsg, "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product/@productId", "363", namespaces ) );
	}
	
	@Test ( expected = XPathExpressionException.class )
	public void shouldThrowIfArgumentIsNotAParsableDouble() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.xmlContentGreaterThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "aaa" ) );
		assertTrue( DslHelper.xmlContentLessThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "aaa" ) );
	}
	
	@Test
	public void xmlContentLessThan() throws UnsupportedEncodingException, XPathExpressionException
	{
		assertTrue( DslHelper.xmlContentLessThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "366" ) );
		assertTrue( DslHelper.xmlContentLessThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "365" ) );
		assertFalse( DslHelper.xmlContentLessThan( msg, "/Order/OrderLines/OrderLine/Product/@productId", "363" ) );
	}
	
	@Test
    public void xmlContentLessThanWithNamespaces() throws UnsupportedEncodingException, XPathExpressionException
    {
        assertTrue( DslHelper.xmlContentLessThan(  nsmsg,  "/ord:Order/ord:OrderLines/ord:OrderLine/p:Product/@productId",  "366", namespaces));
    }
	
	@Test
	public void xmlContentEqualsPerformanceTest() throws UnsupportedEncodingException, XPathExpressionException
	{
		int nrOfCalls = 4000;
		
		long startTime = System.nanoTime();
		for ( int i = 0 ; i < nrOfCalls ; i++ )
		{
    		DslHelper.xmlContentEquals( msg, "/Order/OrderLines/OrderLine/Product/@productId", "364" );
		}
		long endTime = TimeUnit.NANOSECONDS.toMillis( System.nanoTime() - startTime );
		log.info( "Timed " + nrOfCalls + " runs : " + endTime + "ms" );
		assertTrue( nrOfCalls + " of calls should have taken less then 150ms, took " + endTime, endTime < 150 );
	}
	
	@Test 
	public void parseNamespaces()
	{
		Map<String,String> namespaces = DslHelper.parseNamespaces( "pro=http://org.jboss.soa.esb/product, ord=http://org.jboss.soa.esb/order" );
		assertNotNull( namespaces );
		assertEquals( 2, namespaces.size() );
		assertTrue( namespaces.containsKey( "pro" ) );
		assertTrue( namespaces.containsKey( "ord" ) );
	}
	
	@Test ( expected = IllegalArgumentException.class )
	public void parseNamespacesNegative()
	{
		DslHelper.parseNamespaces( null );
	}
	
	@Before
	@After
	public void clearContext()
	{
		RulesContext.clearContext() ;
	}
	
	@BeforeClass
	public static void createMessage() throws UnsupportedEncodingException
	{
		msg = MessageFactory.getInstance().getMessage();
		InputStream resourceAsStream = ClassUtil.getResourceAsStream( "/" + "5KB_message.xml", DslHelperUnitTest.class );
		String contents = StreamUtils.readStreamString( resourceAsStream, "UTF-8" );
		msg.getBody().add( contents );
	}
	
    @BeforeClass
    public static void createNamespaceMessage() throws UnsupportedEncodingException
    {
        nsmsg = MessageFactory.getInstance().getMessage();
        InputStream resourceAsStream = ClassUtil.getResourceAsStream( "/" + "5KBNS_message.xml", DslHelperUnitTest.class );
        String contents = StreamUtils.readStreamString( resourceAsStream, "UTF-8" );
        nsmsg.getBody().add( contents );
        
        namespaces = new HashMap<String,String>();
        namespaces.put( "ord", "http://org.jboss.soa.esb/order" );
        namespaces.put( "p", "http://org.jboss.soa.esb/product" );
    }
    
    public static junit.framework.Test suite() {
        return new JUnit4TestAdapter(DslHelperUnitTest.class);
    }
}
