/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules.util;

import java.util.HashMap;
import java.util.Map;


/**
 * Class providing access to a rules context for an execution.
 */
public class RulesContext 
{
    /**
     * The rules context passed via 
     */
    private static final ThreadLocal<Map<String, Object>> CONTEXT = new InheritableThreadLocal<Map<String,Object>>() ;
    
    /**
     * Clear the context associated with the current thread.
     */
    public static void clearContext()
    {
        CONTEXT.set(null) ;
    }
    
    /**
     * Get the rules context value with the specified name. 
     * @param name The name of the context value.
     * @return The context value or null if not set.
     */
    public static Object getContext(final String name)
    {
        final Map<String, Object> context = CONTEXT.get() ;
        return (context != null) ? context.get(name) : null ;
    }
    
    /**
     * Set the context value.
     * @param name The name of the context value.
     * @param value The associated value or null to remove.
     */
    public static void setContext(final String name, final Object value)
    {
        final Map<String, Object> context = CONTEXT.get() ;
        if (context != null)
        {
            initialiseContext(context, name, value) ;
        }
        else
        {
            final Map<String, Object> newContext = new HashMap<String, Object>() ;
            CONTEXT.set(newContext) ;
            initialiseContext(newContext, name, value) ;
        }
    }
    
    /**
     * Initialise the context value.
     * @param context The current context.
     * @param name The name of the context value.
     * @param value The value of the context or null if it is to be removed.
     */
    private static void initialiseContext(final Map<String, Object> context, final String name, final Object value)
    {
        if (value == null)
        {
            context.remove(name) ;
        }
        else
        {
            context.put(name, value) ;
        }
    }
}
