/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2010
 */
package org.jboss.internal.soa.esb.services.rules.util;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;

/**
 * Holds lifecycle-bound references to whatever the TCCL was at the time of construction,
 * and tries to use that for loading classes or getting resources.  This will not only
 * help us find the correct class or resource, but it will also help us avoid a memory leak,
 * as there are static drools mechanisms (like ResourceChangeScanner/Notifier).
 * 
 * @author dward at jboss.org
 */
public final class RulesClassLoader extends ClassLoader
{
	
	private final String ruleSource;
	
	public RulesClassLoader(final String ruleSource)
	{
		AssertArgument.isNotNullAndNotEmpty(ruleSource, RULE_SOURCE);
		this.ruleSource = ruleSource;
		ClassLoader tccl = Thread.currentThread().getContextClassLoader();
		if (tccl != null)
		{
			try
			{
				putClassLoader(this.ruleSource, tccl);
			}
			catch (LifecycleResourceException lre)
			{
				throw new RuntimeException(lre);
			}
		}
	}
	
	@Override
	public final URL getResource(String name)
	{
		ClassLoader cl = getClassLoader();
		return (cl != null) ? cl.getResource(name) : super.getResource(name);
	}
	
	@Override
	public final InputStream getResourceAsStream(String name)
	{
		ClassLoader cl = getClassLoader();
		return (cl != null) ? cl.getResourceAsStream(name) : super.getResourceAsStream(name);
	}
	
	@Override
	public final Enumeration<URL> getResources(String name) throws IOException
	{
		ClassLoader cl = getClassLoader();
		return (cl != null) ? cl.getResources(name) : super.getResources(name);
	}
	
	@Override
	public final Class<?> loadClass(String name) throws ClassNotFoundException
	{
		ClassLoader cl = getClassLoader();
		return (cl != null) ? cl.loadClass(name) : super.loadClass(name);
	}
	
	@Override
	public final Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException
	{
		ClassLoader cl = getClassLoader();
		return (cl != null) ? cl.loadClass(name) : super.loadClass(name, resolve);
	}
	
	private final ClassLoader getClassLoader()
	{
		try
		{
			return getClassLoader(this.ruleSource);
		}
		catch (LifecycleResourceException lre)
		{
			throw new RuntimeException(lre);
		}
	}
	
	private static final Logger logger = Logger.getLogger(RulesClassLoader.class);
	
	private static final String RULE_SOURCE = "ruleSource";
	
	private static final LifecycleResource<Map<String, ClassLoader>> lifecycleClassLoaders =
		new LifecycleResource<Map<String, ClassLoader>>(new LifecycleRuleAgentFactory(), LifecyclePriorities.RULE_AGENT_RESOURCE_CLASSLOADER_PRIORITY);
	
	private static final void putClassLoader(String ruleSource, ClassLoader classLoader) throws LifecycleResourceException
	{
		String key = buildKey(ruleSource);
		Map<String, ClassLoader> map = lifecycleClassLoaders.getLifecycleResource();
		synchronized (map)
		{
			map.put(key, classLoader);
		}
	}
	
	private static final ClassLoader getClassLoader(String ruleSource) throws LifecycleResourceException
	{
		String key = buildKey(ruleSource);
		Map<String, ClassLoader> map = lifecycleClassLoaders.getLifecycleResource();
		synchronized (map)
		{
			return map.get(key);
		}
	}
	
	private static final String buildKey(String ruleSource)
	{
		return new StringBuilder()
			.append(RulesClassLoader.class.getSimpleName())
			.append("?deployment=")
			.append(LifecycleResourceManager.getSingleton().getIdentity())
			.append('&')
			.append(RULE_SOURCE)
			.append('=')
			.append(ruleSource)
			.toString();
	}
	
	private static final class LifecycleRuleAgentFactory implements LifecycleResourceFactory<Map<String, ClassLoader>>
	{
		
		public Map<String, ClassLoader> createLifecycleResource(String lifecycleIdentity) throws LifecycleResourceException
		{
			return new ConcurrentHashMap<String, ClassLoader>();
		}
		
		public void destroyLifecycleResource(Map<String, ClassLoader> resource, String lifecycleIdentity) throws LifecycleResourceException
		{
			if (resource.size() > 0)
			{
				if ( logger.isInfoEnabled() )
				{
					for ( String key : resource.keySet() )
					{
						logger.info("Removing " + key);
					}
				}
				resource.clear();
			}
		}
        
	}

}
