/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.esb.services.jbpm.integration;

import javax.jms.ConnectionFactory;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jboss.jms.jndi.JMSProviderAdapter;
import org.jbpm.JbpmException;
import org.jbpm.svc.ServiceFactory;

/**
 * Abstract base JMS service factory.
 */
public abstract class AbstractJmsServiceFactory implements ServiceFactory
{
    /**
     * 
     */
    private static final long serialVersionUID = 2312177513288418710L;

    /**
     * The JNDI name of a jms provider adapter.
     */
    String providerAdapterJNDI;
    /**
     * The JNDI name for locating the JMS connection factory.
     */
    String connectionFactoryJndiName = "java:/JmsXA";
    /**
     * A flag indicating whether commit should be invoked on the JMS session.
     */
    boolean isCommitEnabled = false;
    /**
     * The JMS connection factory associated with the scheduler.
     */
    private ConnectionFactory connectionFactory;

    /**
     * The provider.
     */
    private Provider provider ;

    /**
     * Get the commitEnabled flag.
     * @return true if commit is to be invoked on the JMS session, false otherwise.
     */
    public boolean isCommitEnabled()
    {
      return isCommitEnabled;
    }

    /**
     * Lookup a name within JNDI.
     * @param name The JNDI name to locate.
     * @return The object bound to the name.
     * @throws NamingException For errors locating the specified object within JNDI.
     */
    protected Object lookup(String name) throws NamingException
    {
      final Provider provider = getProvider() ;
      final Context initial = provider.getInitialContext() ;
      try
      {
        return initial.lookup(name);
      }
      finally
      {
        initial.close();
      }
    }

    /**
     * Get the provider.
     * @return The provider.
     * @throws NamingException For errors locating the adapter within JNDI.
     */
    protected Provider getProvider()
      throws NamingException
    {
      if (provider == null)
      {
        if (providerAdapterJNDI != null)
        {
          final String lookup ;
          if (!providerAdapterJNDI.startsWith("java:"))
          {
            lookup = "java:" + providerAdapterJNDI ;
          }
          else
          {
            lookup = providerAdapterJNDI ;
          }
          
          final Context initial = new InitialContext();
          final Object adapter ;
          try
          {
            adapter = initial.lookup(lookup);
          }
          finally
          {
            initial.close();
          }
          if (adapter != null)
          {
            if (adapter instanceof JMSProviderAdapter)
            {
              provider = new JMSProvider((JMSProviderAdapter)adapter) ;
            }
            else if (adapter instanceof Provider)
            {
              provider = (Provider)adapter ;
            }
          }
        }
      }
      if (provider == null)
      {
        provider = new DefaultProvider() ;
      }
      return provider ;
    }
    
    /**
     * Get the JMS connection factory associated with this service factory.
     * @return The associated JMS connection factory.
     */
    protected ConnectionFactory getConnectionFactory()
    {
      if (connectionFactory == null)
      {
        try
        {
          final Provider provider = getProvider() ;
          final String jndiName ;
          final String adapterQueueFactory = provider.getQueueFactoryRef() ;
          if (adapterQueueFactory != null)
          {
              jndiName = adapterQueueFactory ;
          }
          else
          {
              final String adapterFactory = provider.getFactoryRef() ;
              if (adapterFactory != null)
              {
                  jndiName = adapterFactory ;
              }
              else
              {
                  jndiName = connectionFactoryJndiName ;
              }
          }
          connectionFactory = (ConnectionFactory)lookup(jndiName);
        }
        catch (NamingException e)
        {
          throw new JbpmException("could not retrieve message connection factory", e);
        }
      }
      return connectionFactory;
    }

    /**
     * Close this factory.
     */
    public void close()
    {
      provider = null;
      connectionFactory = null;
    }
}
