/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.ant;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.taskdefs.MatchingTask;
import org.apache.tools.ant.types.FileSet;

/**
 * ant task for deploying process archives to the deployment servlet.
 * </p>
 * Usage:
 * <pre>
 * &lt;taskdef name="deployToServer" classname="org.jbpm.ant.DeployProcessToServerTask"&gt;
 *     &lt;classpath refid="exec-classpath"/&gt;
 * &lt;/taskdef&gt;
 * 
 * &lt;deployToServer serverDeployer="jbpm-console/upload"&gt;
 *     &lt;fileset dir="${basedir}/processDefinition" includes="*"/&gt;
 * &lt;/deployToServer&gt;
 * </pre>
 * Optional attributes:
 * <pre>
 * serverName="localhost"
 * serverPort="8080"
 * serverDeployer="/jbpm-console/upload"
 * process="processDefinitions/processdefinition.xml"
 * username="username"
 * password="password"
 * </pre>
 * 
 * @author kurt stam
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 */
public class DeployProcessToServerTask extends MatchingTask {

    private static final String boundary = "AaB03x";
    
    String servername = "localhost";
    String serverPort = "8080";
    String serverDeployer = "/jbpm-console/upload";
    String debug = null;
    File process = null;
    List fileSets = new ArrayList();

	private String username;

	private String password;

    public void setServerDeployer(String serverDeployer) {
        this.serverDeployer = serverDeployer;
    }

    public void setServerName(String serverName) {
        this.servername = serverName;
    }

    public void setServerPort(String serverPort) {
        this.serverPort = serverPort;
    }
    
    public void setProcess(File process) {
        this.process = process;
    }
    
    public void setDebug(String debug) {
        this.debug = debug;
    }
    
	public void setUsername( final String userName )
	{
		this.username = userName;
	}

	public void setPassword( final String password )
	{
		this.password = password;
	}

    public void execute() throws BuildException {
        try {
            if (pingServerOK()) {
                //if attribute par is set, deploy the par file
                if (process!=null) {
                    log( "deploying par "+process.getAbsolutePath()+" ..." );
                    FileInputStream in = new FileInputStream(process);
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    byte[] buf = new byte[1024];
                    int len;
                    while ((len = in.read(buf)) > 0) {
                        out.write(buf, 0, len);
                    }
                    out.flush();
                    if (debug!=null) {
                        saveParFile("debug.par", out.toByteArray());
                    }
                    deployProcessWithServlet(servername, serverPort, serverDeployer, out.toByteArray());
                    in.close();
                    log("deployment complete.");
                }
                //if attribute fileSets is set, deploy the fileSets too
                if (fileSets.size()>0) {
                    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
                    ZipOutputStream zipOutputStream = new ZipOutputStream(byteArrayOutputStream);
    
                    // loop over all files that are specified in the filesets
                    Iterator iter = fileSets.iterator();
                    while (iter.hasNext()) {
                        FileSet fileSet = (FileSet) iter.next();
                        DirectoryScanner dirScanner = fileSet.getDirectoryScanner(getProject());
                        String[] fileSetFiles = dirScanner.getIncludedFiles();
    
                        for (int i = 0; i < fileSetFiles.length; i++) {
                            String fileName = fileSetFiles[i];
                            File file = new File(fileName);
                            if ( !file.isFile() ) {
                                file = new File( dirScanner.getBasedir(), fileName );
                            }
                            // deploy the file, specified in a fileset element
                            log( "adding to process archive "+file+" ..." );
                            addFile(zipOutputStream, file);
                        }
                    }
                    zipOutputStream.close();
                    log( "deploying par ..." );
                    if (debug!=null) {
                        saveParFile("debug.par", byteArrayOutputStream.toByteArray());
                    }
                    deployProcessWithServlet(servername, serverPort, serverDeployer, byteArrayOutputStream.toByteArray());
                    log("deployment complete.");
                }
            }
        } catch (Exception e) {
            throw new BuildException( "couldn't deploy process archives : " + e.getMessage(), e);
        }
    }

    public void addFileset(FileSet fileSet) {
        this.fileSets.add(fileSet);
    }

    public void deployProcessWithServlet(final String serverName, final String serverPort, final String serverDeployer, final byte[] parBytes) throws Exception {
        log(deployProcess(serverName, serverPort, serverDeployer, parBytes));
    }
    
    String deployProcess(final String serverName, final String serverPort, final String serverDeployer, final byte[] parBytes) throws Exception {
        String sessionID = null; 
        //	pass security credentials if specified.
        boolean security = verifySecurityProperties();
        if ( security )
        	sessionID = tryFormBasedAuthentication();
        
        URL url = new URL("http://" + serverName + ":" + serverPort + serverDeployer);
        HttpURLConnection urlConnection = (HttpURLConnection)url.openConnection();
        urlConnection.setRequestMethod( "POST" );
        urlConnection.setDoInput(true);
        urlConnection.setDoOutput(true);
        urlConnection.setAllowUserInteraction(false);
        urlConnection.setUseCaches(false);
        
        if ( sessionID != null ) // FORM based authentication
        	urlConnection.setRequestProperty( "Cookie", sessionID  );
        else // fall back to BASIC authentication
            urlConnection.setRequestProperty( "Authorization", "Basic " + encode(username + ":" + password ) );
        	
        urlConnection.setRequestProperty("Content-Type", "multipart/form-data; boundary=AaB03x");
        
        DataOutputStream dataOutputStream = new DataOutputStream(urlConnection.getOutputStream());
        dataOutputStream.writeBytes("--" + boundary + "\r\n");
        dataOutputStream.writeBytes("Content-Disposition: form-data; name=\"definition\"; filename=\"dummy.par\"\r\n");
        dataOutputStream.writeBytes("Content-Type: application/x-zip-compressed\r\n\r\n");

        dataOutputStream.write(parBytes);

        dataOutputStream.writeBytes("\r\n--" + boundary + "--\r\n");
        dataOutputStream.flush();
        dataOutputStream.close();
        
        InputStream inputStream = urlConnection.getInputStream();
        StringBuffer result = new StringBuffer();
        int read;
        while ((read = inputStream.read()) != -1) {
            result.append((char)read);
        }
        return result.toString() ;
    }
    
    /**
     * Adds BASIC Authentication properties to the passed-in urlConnection.
     * 
     * @param urlConnection
     */
    void addBasicSecurityCredentials( final URLConnection urlConnection )
    {
        urlConnection.setRequestProperty("Authorization", "Basic " + encode(username + ":" + password));
    }
    
    /**
     * Try to authenticate using FORM based Authentication. 
     * If succuessful the returned String will contain the JSESSIONID.
     * 
     * @return String	The session id from the serlvet. 
     */
    private String tryFormBasedAuthentication() throws HttpException, IOException
    {
    	final String path ="http://" + servername + ":" + serverPort + serverDeployer;
        String sessionId = null;
        
    	final HttpClient httpClient = new HttpClient();
        GetMethod indexGet = new GetMethod( path );
        
        if ( httpClient.executeMethod(indexGet) == 200 )
        {
        
            HttpState state = httpClient.getState();
            Cookie[] cookies = state.getCookies();
        	for (Cookie cookie : cookies)
			{
                if( cookie.getName().equalsIgnoreCase("JSESSIONID") )
                {
                    sessionId = "JSESSIONID=" + cookie.getValue();
                }
            }
        
            PostMethod loginFormPost = new PostMethod( path + "/j_security_check");
            loginFormPost.addParameter("j_username", username);
            loginFormPost.addParameter("j_password", password);
            httpClient.executeMethod(loginFormPost.getHostConfiguration(), loginFormPost, state);

            Header location = loginFormPost.getResponseHeader("Location");
            String indexURI = location.getValue();
            GetMethod war1Index = new GetMethod(indexURI);
            httpClient.executeMethod(war1Index.getHostConfiguration(), war1Index, state);
        }
        return sessionId;
    }
    
    public void addFile(ZipOutputStream zos, File file)
    throws IOException 
    {
        byte[] buff = new byte[256];
        if (!file.exists()) return;
        InputStream is = new FileInputStream(file);
        zos.putNextEntry(new ZipEntry(file.getName()));
        int read;
        while ((read = is.read(buff)) != -1) {
            zos.write(buff, 0, read);
        }
    }
    
    private void saveParFile(String fileName, byte[] parBytes) throws IOException {
        File file = new File(fileName);
        if (!file.exists()) {
            file.createNewFile();
        }
        FileOutputStream fos = new FileOutputStream(file);
        fos.write(parBytes);
        fos.close();
    }
    
    public boolean pingServerOK() {
        URL url = null;
        try {
            url = new URL("http://" + servername + ":" + serverPort + serverDeployer);
            URLConnection urlConnection = url.openConnection();
            urlConnection.setDoOutput(true);
            //	pass security credentials is specified.
            if ( verifySecurityProperties() )
                addBasicSecurityCredentials( urlConnection );
            InputStream inputStream = urlConnection.getInputStream();
            StringBuffer result = new StringBuffer();
            int read;
            while ((read = inputStream.read()) != -1) {
                result.append((char)read);
            }
            return true;
        }
        catch (ConnectException e) {
            log("The server (" + url + ") could not be reached.");
            return false;
        }
        catch (Exception e) {
        	e.printStackTrace();
            log("An unexpected exception happened while testing the server (" + url + ") connection.");
            return false;
        }
    }

	boolean verifySecurityProperties() throws BuildException
	{
		if ( username == null && password == null || "".equals( username ) && "".equals( password ) )
			return false;
		
		if ( username != null && password == null )
			throw new BuildException( "'userName' must be accompanied with 'password'");
		else if ( password != null && username == null )
			throw new BuildException( "'password' must be accompanied with 'userName'");
		
		return true;
	}
	
    public static String encode (String source) {
        return new String( new Base64().encode( source.getBytes() ));
    }
}