/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author daniel.brum@jboss.com
 */

package org.jboss.soa.esb.persistence.tests;

/**
 * @author dbrum
 * 
 * 
 */
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.URI;
import java.sql.Connection;
import java.sql.Statement;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;
import org.jboss.internal.soa.esb.persistence.format.MessageStoreFactory;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.persistence.manager.ConnectionManager;
import org.jboss.soa.esb.persistence.manager.ConnectionManagerException;
import org.jboss.soa.esb.persistence.manager.ConnectionManagerFactory;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

public class MessageStoreUnitTest
{

	private static Logger log = Logger.getLogger(MessageStoreUnitTest.class);
	
	private static Object server ;
	
    @Test
	public void testDefaultMessageStore ()
	{
		MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
		assertEquals((store != null), true);
	}

    @Test
	public void testURIMessageStore ()
	{
		// only DB supported for now
		MessageStore store1 = MessageStoreFactory.getInstance().getMessageStore();
		assertEquals((store1 != null), true);
	}
	
    @Test
	public void testSettingUndeliveredDelivered () throws Exception
	{
		//test setting one message to undelivered status
//		 add new messages
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
		assertNotNull("Message", msg);
		
		MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
		assertNotNull("store", store);
		
		Map<URI, Message> messages = store.getUndeliveredMessages(MessageStore.CLASSIFICATION_DEFAULT) ;
		assertEquals("Initial undelivered count", 0, messages.size()) ;
		
		URI uid = store.addMessage(msg, MessageStore.CLASSIFICATION_DEFAULT);
		
		messages = store.getUndeliveredMessages(MessageStore.CLASSIFICATION_DEFAULT) ;
		assertEquals("Undelivered count after addition", 0, messages.size()) ;
		
		store.setUndelivered(uid);

		messages = store.getUndeliveredMessages(MessageStore.CLASSIFICATION_DEFAULT) ;
		assertEquals("Undelivered count after setUndelivered", 1, messages.size()) ;
		
		store.setDelivered(uid);

		messages = store.getUndeliveredMessages(MessageStore.CLASSIFICATION_DEFAULT) ;
		assertEquals("Undelivered count after setDelivered", 0, messages.size()) ;
		
		final Message storedMessage = store.getMessage(uid);
		assertNotNull("Stored message", storedMessage) ;
	}
	
    @Test
	public void testManyUndeliveredMessages() throws Exception {
		Message msg1, msg2, msg3, msg4;
		msg1 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);
		msg2 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);
		msg3 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);
		msg4 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);
		
		assertEquals((msg1 != null), true);
		assertEquals((msg2 != null), true);
		assertEquals((msg3 != null), true);
		assertEquals((msg4 != null), true);


		MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
		assertEquals((store != null), true);
		
		URI uid1 = store.addMessage(msg1,MessageStore.CLASSIFICATION_DEFAULT);
		URI uid2 = store.addMessage(msg2,MessageStore.CLASSIFICATION_DEFAULT);
		URI uid3 = store.addMessage(msg3,MessageStore.CLASSIFICATION_DEFAULT);
		URI uid4 = store.addMessage(msg4,MessageStore.CLASSIFICATION_DEFAULT);
		
		store.setUndelivered(uid1);
		store.setUndelivered(uid2);
		store.setUndelivered(uid3);
		store.setUndelivered(uid4);
		
		Map<URI, Message> messages = store.getUndeliveredMessages(MessageStore.CLASSIFICATION_DEFAULT);
		assertEquals((messages != null), true);

	}

    @Test
	public void testStoreAndRetrieve () throws Exception
	{
		// add new messages
		Message msg1 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);
		Message msg2 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);
		assertEquals((msg1 != null), true);
		assertEquals((msg2 != null), true);

		MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
		assertEquals((store != null), true);

		// set some properties inside the Message
		msg1.getProperties().setProperty("prop1", "val1");
		msg2.getProperties().setProperty("prop1", "val1");
		msg1.getProperties().setProperty("prop2", "val2");
		msg2.getProperties().setProperty("prop2", "val2");

		// set the body inside the Message
		msg1.getBody().add(("TEST BODY").getBytes());
		msg2.getBody().add(("TEST BODY").getBytes());

		// set some object attachments inside the Message
		msg1.getAttachment().addItem(new String("TEST ATTACHMENT1"));
		msg1.getAttachment().addItem(new String("TEST ATTACHMENT2"));
		msg2.getAttachment().addItem(new String("TEST ATTACHMENT1"));
		msg2.getAttachment().addItem(new String("TEST ATTACHMENT2"));

		// keep track of the UID to use in retrieving the Message
		URI uid1 = null;
		uid1 = store.addMessage(msg1, MessageStore.CLASSIFICATION_DEFAULT);
//		System.out.println(uid1);
		assertEquals((uid1 != null), true);

		URI uid2 = null;
		uid2 = store.addMessage(msg2,MessageStore.CLASSIFICATION_DEFAULT);
//		System.out.println(uid2);
		assertEquals((uid2 != null), true);

		// now retrieve the messages
		try
		{
			Message msgIn1 = store.getMessage(uid1);
			assertEquals((msgIn1 != null), true);

			Message msgIn2 = store.getMessage(uid2);
			assertEquals((msgIn2 != null), true);
		}
		catch (Exception e)
		{
			log.error( e );
		}
	}

    @Test
	public void testDBConnectionManager ()
	{
		MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
		assertEquals((store != null), true);
		ConnectionManager mgr=null;
		try {
			mgr = ConnectionManagerFactory.getConnectionManager();
		} catch (ConnectionManagerException e) {
			// TODO Auto-generated catch block
//			e.printStackTrace();
		}
		assertEquals((mgr != null), true);
	}

    @BeforeClass
    public static void runBeforeAllTests()
    {
        try {
            File testResourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/test/resources/");
            System.out.println("Current dir=" + testResourceDir.getCanonicalPath());
            DOMConfigurator.configure(testResourceDir.getCanonicalPath() + "/log4j.xml");
            File buildDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/build/");
            File resourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/main/resources/");
            System.setProperty("org.jboss.soa.esb.propertyFile", "jbossesb-unittest-properties.xml");                    
            if ("org.hsqldb.jdbcDriver".equals(Configuration.getStoreDriver())) {
                final String databaseFile = buildDir + "/hsqltestdb";
                HsqldbUtil.dropHsqldb(databaseFile);
                server = HsqldbUtil.startHsqldb(databaseFile, "jbossesb");
                
                //Drop what is there now, if exists. We want to start fresh.                
                String sqlCreateCmd    = TestEnvironmentUtil.readTextFile(new File(resourceDir.getCanonicalPath() + "/message-store-sql/hsqldb/create_database.sql"));
                String sqlDropCmd      = TestEnvironmentUtil.readTextFile(new File(resourceDir.getAbsolutePath() + "/message-store-sql/hsqldb/drop_database.sql"));
                
                ConnectionManager mgr = ConnectionManagerFactory.getConnectionManager();
                mgr.init();
                Connection con = mgr.getConnection();
                Statement stmnt = con.createStatement();
                System.out.println("Dropping the schema if exist");
                stmnt.execute(sqlDropCmd);
                System.out.println("Creating the message store schema");
                stmnt.execute(sqlCreateCmd);
            }
        } catch (Throwable e) {
            e.printStackTrace();
            System.out.println("We should stop testing, since we don't have a db.");
            assertTrue(false);
        }
        
    }
	@AfterClass
	public static void runAfterAllTests ()
	{
        
		try
		{
			if (Configuration.getStoreDriver().equals("org.hsqldb.jdbcDriver"))
				HsqldbUtil.stopHsqldb(server);
		}
		catch (Exception e)
		{ // 
			log.error( e );
		}
	}

	/**
	 * Read the file into a String.
	 * 
	 * @param file -
	 *            the file to be read
	 * @return String with the content of the file
	 * @throws IOException -
	 *             when we can't read the file
	 */
	public static String readTextFile (File file) throws IOException
	{
		StringBuffer sb = new StringBuffer(1024);
		BufferedReader reader = new BufferedReader(new FileReader(file
				.getPath()));
		char[] chars = new char[1];
		while ((reader.read(chars)) > -1)
		{
			sb.append(String.valueOf(chars));
			chars = new char[1];
		}
		reader.close();
		return sb.toString();
	}

	public static junit.framework.Test suite ()
	{
		return new JUnit4TestAdapter(MessageStoreUnitTest.class);
	}

}
