/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

/**
 * Routes the Message argument to a fixed list of services ([category,name]) 
 * @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 * @since Version 4.0
 */
package org.jboss.soa.esb.actions;

import java.net.URI;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.persistence.format.MessageStoreFactory;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.services.persistence.MessageStoreException;

public class MessagePersister extends AbstractActionPipelineProcessor
{
    public final static String MESSAGE_STORE_CLASS = "message-store-class";
    public final static String CLASSIFICATION_ATTR = "classification";
    public static final String TERMINAL = "terminal";
    
    protected ConfigTree config;
    protected MessageStore messageStore;
    protected String classification;
    protected boolean terminal = true;
    
	private Logger log = Logger.getLogger(this.getClass());

	public MessagePersister(ConfigTree config) throws ConfigurationException
	{
        this.config = config;
	}
	/** 
     * Persists the message to the MessageStore.
     *@return if this is a terminal action then <code>null</code> otherwise the original
     * input message.
	 */
	public Message process(Message message) throws ActionProcessingException
	{
        String classification = this.classification;
        try {
            //the message can override the classification
            if (message.getProperties().getProperty(MessageStore.CLASSIFICATION)!=null) {
                classification = String.valueOf(message.getProperties().getProperty(MessageStore.CLASSIFICATION));
                message.getProperties().remove(MessageStore.CLASSIFICATION);
            }
            URI uri = messageStore.addMessage(message, classification);
            message.getProperties().setProperty(MessageStore.MESSAGE_URI, uri);
        } catch (MessageStoreException mse) {
            log.error("Could not store message.", mse);
        }
        
        if (terminal)
		return null;
        else
            return message;
	}
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionLifecycle#initialise()
     */
    public void initialise() throws ActionLifecycleException {
        //Reading the config
        String messageStoreClass = config.getAttribute(MESSAGE_STORE_CLASS);
        String classificationValue   = config.getAttribute(CLASSIFICATION_ATTR);
        if (classificationValue!=null) {
            classification = classificationValue;
        }

        messageStore = MessageStoreFactory.getInstance().getMessageStore(messageStoreClass, config);
        
        if ("false".equalsIgnoreCase(config.getAttribute(TERMINAL)))
            terminal = false;
        
        if (log.isDebugEnabled()) {
            log.debug("MessagePersister started with classification=" + classification 
                + " and message-store-class=" + messageStore);
        }
    }
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionLifecycle#destroy()
     */
    public void destroy() throws ActionLifecycleException {
       classification=null;
    }
    
    public Message getMessages(Message message) throws ActionProcessingException
    {
        String classification = this.classification;
        try {
            //the message can override the classification
            if (message.getProperties().getProperty(MessageStore.CLASSIFICATION)!=null) {
                classification = String.valueOf(message.getProperties().getProperty(MessageStore.CLASSIFICATION));
            }
            Map<URI, Message> messageMap = messageStore.getAllMessages(classification);
            message.getBody().add(messageMap);
        } catch (MessageStoreException mse) {
            log.error("Could obtain messages.", mse);
        }
        return message;
    }
    
    public Message getMessage(Message message) throws ActionProcessingException
    {
        if (message.getProperties().getProperty(MessageStore.MESSAGE_URI)!=null) {
            URI uid = (URI) message.getProperties().getProperty(MessageStore.MESSAGE_URI);
            try {
                //the message can override the classification
                message = messageStore.getMessage(uid);
            } catch (MessageStoreException mse) {
                log.error("Could obtain messages.", mse);
            } 
        } else {
            log.error("Required property " + MessageStore.MESSAGE_URI + " is missing");
        }
        return message;
    }
    
    public Message removeMessage(Message message) throws ActionProcessingException
    {
        String classification = this.classification;
        if (message.getProperties().getProperty(MessageStore.MESSAGE_URI)!=null) {
            URI uid = (URI) message.getProperties().getProperty(MessageStore.MESSAGE_URI);
            try {
                if (message.getProperties().getProperty(MessageStore.CLASSIFICATION)!=null) {
                    classification = String.valueOf(message.getProperties().getProperty(MessageStore.CLASSIFICATION));
                }
                //the message can override the classification
                messageStore.removeMessage(uid, classification);
            } catch (MessageStoreException mse) {
                log.error("Could obtain messages.", mse);
            } 
        } else {
            log.error("Required property " + MessageStore.MESSAGE_URI + " is missing");
        }
        return message;
    }
    
}
