/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URL;
import java.util.HashSet;
import java.util.concurrent.TimeUnit;

import javax.crypto.SealedObject;
import javax.security.auth.Subject;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.principals.Group;
import org.jboss.soa.esb.services.security.principals.Role;
import org.jboss.soa.esb.services.security.principals.User;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * Unit test for SecurityContext.
 * <p/>
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class SecurityContextUnitTest
{
	private String jbossEsbProperties;

    @Test
	public void isCallerInRole()
	{
		Subject subject = new Subject();
		subject.getPrincipals().add( new User("AustinPowerwich") );
		Group roles = new Group("Roles");
		roles.addMember( new Role("Admin"));
		subject.getPrincipals().add(roles);

		SecurityContext securityContext = new SecurityContext(subject, 30000);
		boolean callerInRole = securityContext.isCallerInRole("Admin");
		assertTrue( callerInRole );
	}

	@Test
    public void serializeAndDeserialize() throws IOException, ClassNotFoundException
    {
        Subject subject = new Subject();
        User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        byte[] publicCred = "secret".getBytes();
        subject.getPublicCredentials().add(publicCred);

        SecurityContext securityContext = new SecurityContext(subject, 30000l);
        assertEquals( user, securityContext.getSubject().getPrincipals().iterator().next() );
        assertEquals( publicCred, securityContext.getSubject().getPublicCredentials().iterator().next());

        //  serialize object
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(bout);
        out.writeObject(securityContext);

        //  deserialize object
        ByteArrayInputStream bin = new ByteArrayInputStream(bout.toByteArray());
        ObjectInputStream oin = new ObjectInputStream(bin);
        Object readObject = oin.readObject();

        //  assert that the content is still matches.
        assertTrue (readObject instanceof SecurityContext);
        SecurityContext deserialized = (SecurityContext)readObject;
        assertEquals( user, deserialized.getSubject().getPrincipals().iterator().next() );
    }

	@Test
    public void encryptThenSerializeAndDeserialize() throws IOException, ClassNotFoundException, SecurityServiceException
    {
        Subject subject = new Subject();
        User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        byte[] publicCred = "secret".getBytes();
        subject.getPublicCredentials().add(publicCred);

        SecurityContext securityContext = new SecurityContext(subject, 30000l);
        Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        SealedObject sealedObject = SecurityContext.encryptContext(securityContext);
        message.getContext().setContext(SecurityService.CONTEXT, sealedObject);

        //  serialize object
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(bout);
        out.writeObject(message);

        //  deserialize object
        ByteArrayInputStream bin = new ByteArrayInputStream(bout.toByteArray());
        ObjectInputStream oin = new ObjectInputStream(bin);
        Object readObject = oin.readObject();

        //  assert that the content is still matches.
        assertTrue (readObject instanceof Message);
        Message deserializedMsg = (Message) readObject;
        SecurityContext deserialized = SecurityContext.decryptContext((SealedObject) deserializedMsg.getContext().getContext(SecurityService.CONTEXT));
        assertEquals( user, deserialized.getSubject().getPrincipals().iterator().next() );
    }

	@Test
	public void decryptionConstructor() throws SecurityServiceException
	{
        Subject subject = new Subject();
        User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        byte[] publicCred = "publicsecret".getBytes();
        subject.getPublicCredentials().add(publicCred);
        byte[] privateCred = "privatesecret".getBytes();
        subject.getPrivateCredentials().add(privateCred);

        SecurityContext securityContext = new SecurityContext(subject, 30000);
        SealedObject sealedObject = SecurityContext.encryptContext(securityContext);
        assertNotNull(sealedObject);

        SecurityContext decryptContext = SecurityContext.decryptContext(sealedObject);
        assertEquals( user, decryptContext.getSubject().getPrincipals().iterator().next());
	}

	@Test
    public void compareTo() throws SecurityServiceException
    {
        Subject subject = new Subject();
        User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        byte[] publicCred = "publicsecret".getBytes();
        subject.getPublicCredentials().add(publicCred);
        byte[] privateCred = "privatesecret".getBytes();
        subject.getPrivateCredentials().add(privateCred);

        SecurityContext securityContext = new SecurityContext(subject, 30000);

        HashSet<Object> credentials = new HashSet<Object>();
        credentials.add(publicCred);
        credentials.add(privateCred);

        AuthenticationRequest authRequest = new AuthenticationRequestImpl.Builder(user, credentials).build();
        assertTrue(securityContext.compareTo(authRequest));

        credentials = new HashSet<Object>();
        credentials.add("public-modified-secret".getBytes());
        authRequest = new AuthenticationRequestImpl.Builder(user, credentials).build();
        assertFalse(securityContext.compareTo(authRequest));
        assertFalse(securityContext.compareTo(null));
    }

	@Test
	public void timeout() throws InterruptedException
	{
        final Subject subject = new Subject();
        SecurityContext context = new SecurityContext(subject, 30000);
        assertTrue("Default is 5 min so context should be valid", context.isValid());

        context = new SecurityContext(subject, 100);
        TimeUnit.SECONDS.sleep(1);
        assertFalse(context.isValid());

        context = new SecurityContext(subject, 1000);
        assertTrue(context.isValid());
	}

	@Test
    public void constructWithNeverExpiringContext()
    {
        SecurityContext securityContext = new SecurityContext(new Subject(), -1);
        assertTrue(securityContext.isValid());
    }

	@Test (expected = IllegalArgumentException.class )
    public void shouldThrowIfTimeoutIsNegative()
    {
        new SecurityContext(new Subject(), -2);
    }

	@Test
    public void getConfigurationTimeout() throws SecurityServiceException
    {
        final long timeout = SecurityContext.getConfigurationTimeout();
        assertEquals("timeout should match the value in security-properties.xml", 30000l, timeout);
    }

	@Before
    public void setup() throws ConfigurationException
    {
        jbossEsbProperties = System.getProperty(Environment.PROPERTIES_FILE);
        URL resource = ClassUtil.getResource("security-properties.xml", getClass());
        System.setProperty(Environment.PROPERTIES_FILE, "abs://" + resource.getFile());
    }

    @After
    public void tearDown()
    {
        if ( jbossEsbProperties != null )
            System.setProperty(Environment.PROPERTIES_FILE, jbossEsbProperties);
    }

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(SecurityContextUnitTest.class);
	}
}
