/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import static org.jboss.soa.esb.services.security.SecurityConfigTestUtil.createSecurityFragment;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.junit.Test;

/**
 * Unit test for {@link SecurityConfigUtil}
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class SecurityConfigUtilUnitTest
{
	private final String runAs = "kalle";

	@Test
	public void createSecurityConfigInfoWithAuth()
	{
		final ConfigTree securityFragment = createSecurityFragment(runAs, null, null);

		SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
		assertEquals(runAs, securityInfo.getRunAs());
		assertNull(securityInfo.getUseCallerIdentity());
	}

	@Test
	public void createSecurityConfigInfoWithoutAuth()
	{
		final ConfigTree securityFragment = createSecurityFragment(runAs, null, null);
		SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
		assertEquals(runAs, securityInfo.getRunAs());
		assertNull(securityInfo.getUseCallerIdentity());
	}

	@Test
	public void createSecurityConfigInfoWithUseCallersIdentity()
	{
		final String callersIdentity = "DrCox";
		final ConfigTree securityFragment = createSecurityFragment(runAs, callersIdentity, null);
		SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
		assertEquals(callersIdentity, securityInfo.getUseCallerIdentity());
	}

	@Test
	public void createSecurityConfigInfoWithModuleName()
	{
		final String moduleName = "testModule";
		final ConfigTree securityFragment = createSecurityFragment(runAs, null, moduleName);
		SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
		assertEquals(moduleName, securityInfo.getModuleName());
	}

	@Test
	public void hasRunAs()
	{
		final ConfigTree securityFragment = createSecurityFragment(runAs, null, null);
		SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
		assertTrue(securityInfo.hasRunAs());
	}

	@Test
	public void properties()
	{
		final ConfigTree securityFragment = createSecurityFragment(runAs, null, null, "kalle");
		SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
		Map<String, String> properties = securityInfo.getProperties();
		assertNotNull( properties );
		assertEquals("kalle", properties.get("alias"));
	}

	@Test
    public void rolesAllowed()
    {
        final ConfigTree securityFragment = createSecurityFragment(runAs, "role1, role2, ", null, null, null );
        SecurityConfig securityInfo = SecurityConfigUtil.createSecurityConfig(securityFragment);
        assertTrue(securityInfo.getRolesAllowed().contains("role1"));
        assertTrue(securityInfo.getRolesAllowed().contains("role2"));
    }

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(SecurityConfigUtilUnitTest.class);
	}
}
