/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;

import junit.framework.TestCase;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;

/**
 * NotifyFiles unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class NotifyFilesUnitTest extends TestCase {
	
	public void test_BadFolderInfo() throws Exception {
		ConfigTree rootEl = new ConfigTree("notif");
		ConfigTree fileEl = new ConfigTree("file",rootEl);
			
		try {
			new NotifyFiles(rootEl);
		} catch(IllegalArgumentException e) {
			// Expected
		}
		fileEl.setAttribute("URI", new File("./").toURI().toString() + "/xyzzyx/a.txt");
		try {
			new NotifyFiles(rootEl);
		} catch(IllegalArgumentException e) {
			// Expected
		}
	}
	
	public void test_StringObj() throws Exception {
		sendNotification("Hello");
	}

	public void test_NonStringObj() throws Exception {
		sendNotification(new Integer(1234));
	}
	
	public void sendNotification(Serializable obj) throws Exception {
		ConfigTree rootEl = new ConfigTree("notif");
		NotifyFiles notifyFiles;
		
		// Add the file configs to the config DOM for 4 files...
		addFileConfig(rootEl, "file1.notif", true);
		addFileConfig(rootEl, "file2.notif", true);
		addFileConfig(rootEl, "file3.notif", false);
		addFileConfig(rootEl, "file4.notif", null);
		
		// Create the class and call the sendNotification method twice...
		notifyFiles = new NotifyFiles(rootEl);
        Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        message.getBody().add("object".getBytes());
		notifyFiles.sendNotification(message);
		notifyFiles.sendNotification(message);

		// Test the files. Files 1 and 2 should be the same size. Files 3 and 4 should be the
		// same size. Files 1 and 2 should be twice as big as files 3 and 4 because append was set
		// "true" on 1 and 2 and sendNotification() was called twice... 
		File file1Obj = getFileObject("file1.notif");
		File file2Obj = getFileObject("file2.notif");
		File file3Obj = getFileObject("file3.notif");
		File file4Obj = getFileObject("file4.notif");
		assertEquals(file1Obj.length(), file2Obj.length());
		assertEquals(file3Obj.length(), file4Obj.length());
		assertEquals(file1Obj.length(), file3Obj.length() * 2);
	}
	
	public void test_sendNotficationNegative()
	{
		final ConfigTree rootEl = new ConfigTree("notif");
		addFileConfig(rootEl, "file1.notif", true);
		addFileConfig(rootEl, "file2.notif", true);
		final NotifyFiles notifyFiles = new MockNotifyFiles(rootEl);
        final Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        message.getBody().add("object".getBytes());
		try
		{
			notifyFiles.sendNotification( message );
			fail( "sendNotification should have thrown a NotificationException");
		}
		catch(Exception e)
		{
			assertTrue ( e instanceof NotificationException );
		}
		
	}
	
	private void addFileConfig(ConfigTree rootEl, String file, Boolean append) {
		File fileObj = getFileObject(file);
		ConfigTree fileEl = new ConfigTree("file",rootEl);
		
		fileEl.setAttribute("URI", fileObj.toURI().toString());
		fileEl.setAttribute("append", (append != null?append.toString():null));
		
		// Also want to delete the file incase the tests are being rerun...
		fileObj.delete();
	}

	private File getFileObject(String file) {
		// TF: Excuse this hack!!!  It's the safest way of calculating a 
		// relative URI within the test structure.  It will prob cause
		// Maven some issues i.e. getting it to work for both Ant and Maven :-(
		// This code assumes that the working dir is the "product" folder.
		
		File fileObj = new File("build/tests/" + getPKg() + "/" + file);
		if(!fileObj.getParentFile().exists()) {
			// The build folder may not exist e.g. if running this test from
			// inside your IDE.
			fileObj.getParentFile().mkdirs();
		}
		
		return fileObj;
	}

	private String getPKg() {
		String javaPackage = NotifyFilesUnitTest.class.getPackage().getName();
		return javaPackage.replace('.', '/');
	}
	
	private static class MockNotifyFiles extends NotifyFiles
	{

		public MockNotifyFiles(ConfigTree p_oP)
		{
			super( p_oP );
		}
		
		protected void stringNotification (FileOutputStream p_oF, String p_s) throws IOException
		{
			throw new IOException("Mock IOException from stringNotification...");
		} // __________________________________
		
		protected void objectNotification (FileOutputStream p_oF, Object p_o) throws IOException
		{
			throw new IOException("Mock IOException from objectNotification...");
		} // __________________________________
		
	}
}
