/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.tests;

import java.io.Serializable;

import org.jboss.internal.soa.esb.message.format.serialized.SerializedValueImpl;

import junit.framework.TestCase;

/**
 * Unit tests for checking serialised wrapping of values.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */

public class SerializedValueUnitTest extends TestCase
{
    public void testObjectTypes()
    {
        // Objects
        verifyEquality("Null value", null) ;
        verifyEquality("Boolean type", Boolean.TRUE) ;
        verifyEquality("Byte type", Byte.valueOf((byte)1)) ;
        verifyEquality("Short type", Short.valueOf((short)2)) ;
        verifyEquality("Character type", Character.valueOf((char)3)) ;
        verifyEquality("Integer type", Integer.valueOf(4)) ;
        verifyEquality("Long type", Long.valueOf(5)) ;
        verifyEquality("Float type", Float.valueOf(6)) ;
        verifyEquality("Double type", Double.valueOf(7)) ;
        verifyEquality("String type", "8") ;
        final SerializedValueImpl wrapped = (SerializedValueImpl)SerializedValueImpl.wrap(new NonNativeType()) ;
        verifyEquality("Wrapped type", wrapped) ;
    }
    
    public void testBaseTypeArrays()
    {
        // base type arrays
        verifyEquality("boolean array type", new boolean[] {true, false}) ;
        verifyEquality("byte array type", new byte[] {0, 1}) ;
        verifyEquality("short array type", new short[] {2, 3}) ;
        verifyEquality("char array type", new char[] {4, 5}) ;
        verifyEquality("int array type", new int[] {6, 7}) ;
        verifyEquality("long array type", new long[] {8, 9}) ;
        verifyEquality("float array type", new float[] {10, 11}) ;
        verifyEquality("double array type", new double[] {12, 13}) ;
    }
    
    public void testBaseTypeNestedArrays()
    {
        // base type nested arrays
        verifyEquality("boolean nested array type", new boolean[][] {{true, false}, {true, false}}) ;
        verifyEquality("byte nested array type", new byte[][] {{0, 1}, {0, 1}}) ;
        verifyEquality("short nested array type", new short[][] {{2, 3}, {2, 3}}) ;
        verifyEquality("char nested array type", new char[][] {{4, 5}, {4, 5}}) ;
        verifyEquality("int nested array type", new int[][] {{6, 7}, {6, 7}}) ;
        verifyEquality("long nested array type", new long[][] {{8, 9}, {8, 9}}) ;
        verifyEquality("float nested array type", new float[][] {{10, 11}, {10, 11}}) ;
        verifyEquality("double nested array type", new double[][] {{12, 13}, {12, 13}}) ;
    }
    
    public void testObjectTypeArrays()
    {
        // object type arrays
        verifyEquality("Boolean array type", new Boolean[] {Boolean.TRUE}) ;
        verifyEquality("Byte array type", new Byte[] {Byte.valueOf((byte)1)}) ;
        verifyEquality("Short array type", new Short[] {Short.valueOf((short)2)}) ;
        verifyEquality("Character array type", new Character[] {Character.valueOf((char)3)}) ;
        verifyEquality("Integer array type", new Integer[] {Integer.valueOf(4)}) ;
        verifyEquality("Long array type", new Long[] {Long.valueOf(5)}) ;
        verifyEquality("Float array type", new Float[] {Float.valueOf(6)}) ;
        verifyEquality("Double array type", new Double[] {Double.valueOf(7)}) ;
        verifyEquality("String array type", new String[] {"8"}) ;
        final SerializedValueImpl wrapped = (SerializedValueImpl)SerializedValueImpl.wrap(new NonNativeType()) ;
        verifyEquality("Wrapped type", new SerializedValueImpl[] {wrapped}) ;
    }
    
    public void testObjectTypeNestedArrays()
    {
        // object type nested arrays
        verifyEquality("Boolean nested array type", new Boolean[][] {{Boolean.TRUE}, {Boolean.TRUE}}) ;
        verifyEquality("Byte nested array type", new Byte[][] {{Byte.valueOf((byte)1)}, {Byte.valueOf((byte)1)}}) ;
        verifyEquality("Short nested array type", new Short[][] {{Short.valueOf((short)2)}, {Short.valueOf((short)2)}}) ;
        verifyEquality("Character nested array type", new Character[][] {{Character.valueOf((char)3)}, {Character.valueOf((char)3)}}) ;
        verifyEquality("Integer nested array type", new Integer[][] {{Integer.valueOf(4)}, {Integer.valueOf(4)}}) ;
        verifyEquality("Long nested array type", new Long[][] {{Long.valueOf(5)}, {Long.valueOf(5)}}) ;
        verifyEquality("Float nested array type", new Float[][] {{Float.valueOf(6)}, {Float.valueOf(6)}}) ;
        verifyEquality("Double nested array type", new Double[][] {{Double.valueOf(7)}, {Double.valueOf(7)}}) ;
        verifyEquality("String nested array type", new String[][] {{"8"}, {"8"}}) ;
        final SerializedValueImpl wrapped = (SerializedValueImpl)SerializedValueImpl.wrap(new NonNativeType()) ;
        verifyEquality("Wrapped type", new SerializedValueImpl[][] {{wrapped}, {wrapped}}) ;
    }
    
    public void testNonNativeType()
    {
        verifyWrapped("Non-native type", new NonNativeType()) ;
    }
    
    private void verifyEquality(final String message, final Serializable value)
    {
        final Serializable serializable = SerializedValueImpl.wrap(value) ;
        if (value != null)
        {
            assertNotNull(message, serializable) ;
        }
        assertEquals(message, value, serializable) ;
    }

    private void verifyWrapped(final String message, final Serializable value)
    {
        final Serializable serializable = SerializedValueImpl.wrap(value) ;
        assertNotNull(message, serializable) ;
        assertEquals(message, SerializedValueImpl.class, serializable.getClass()) ;
    }

    private class NonNativeType implements Serializable
    {
        private static final long serialVersionUID = 6703031480417642685L;
    }
}
