/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.tests;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URI;

import junit.framework.TestCase;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.message.format.serialized.MessageImpl;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.EmailEpr;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FTPSEpr;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.addressing.eprs.HTTPEpr;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.addressing.eprs.SFTPEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;

/**
 * Unit tests for the Class class.
 * 
 * @author Mark Little
 */

public class SerializedMessageUnitTest extends TestCase
{
	private Logger log = Logger.getLogger( SerializedMessageUnitTest.class );

	public void testSerialize()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);

		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
		
	}

	public void testDeserialize()
	{
		// get XML message

		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);

		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
			
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			@SuppressWarnings("unused")
			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}
	
	public void testHeader ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);

		Call call = new Call();

		msg.getHeader().setCall(call);
		
		call = msg.getHeader().getCall();
		
		assertEquals((call != null), true);
		
		try
		{
			msg.getHeader().setCall(null);
			
			fail();
		}
		catch (IllegalArgumentException ex)
		{
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}
	
	public void testInvalidAdd ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);
		
		try
		{
			msg.getBody().add(null, null);
			
			fail();
		}
		catch (IllegalArgumentException ex)
		{
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}

	public void testAddBytes ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);
		
		String testString = "test";
		
		msg.getBody().add(BytesBody.BYTES_LOCATION, testString.getBytes());
		
		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
			
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			String val = new String((byte[]) nImpl.getBody().get(BytesBody.BYTES_LOCATION));
			
			assertEquals(val, testString);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testReplace ()
	{
		Message msg1 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg1 != null), true);
		
		String foo = "foo";
		
		Message msg2 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg2 != null), true);
		
		String bar = "bar";
		
		msg1.getBody().add(BytesBody.BYTES_LOCATION, foo.getBytes());
		msg2.getBody().add(BytesBody.BYTES_LOCATION, bar.getBytes());
		
		msg1.getBody().replace(msg2.getBody());
		
		String foobar = new String((byte[]) msg1.getBody().get(BytesBody.BYTES_LOCATION));
		
		assertEquals(foobar.equals("bar"), true);
	}
	
	public void testMerge ()
	{
		Message msg1 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg1 != null), true);
		
		String foo = "foo";
		
		Message msg2 = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg2 != null), true);
		
		String bar = "bar";
		
		msg1.getBody().add(BytesBody.BYTES_LOCATION, foo.getBytes());
		
		assertEquals(new String((byte[]) msg1.getBody().get(BytesBody.BYTES_LOCATION)), "foo");
		
		msg2.getBody().add(BytesBody.BYTES_LOCATION, bar.getBytes());
		
		msg1.getBody().merge(msg2.getBody());
		
		String foobar = new String((byte[]) msg1.getBody().get(BytesBody.BYTES_LOCATION));
		
		assertEquals(foobar, "bar");
	}
	
	public void testAddObjects ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);
		
		ExampleObject value = new ExampleObject(1234);
		
		msg.getBody().add("foo", value);
		
		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
			
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			ExampleObject foo = (ExampleObject) nImpl.getBody().get("foo");
			
			assertEquals((foo.getValue() == value.getValue()), true);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testGetNames ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);
		
		ExampleObject value = new ExampleObject(1234);
		
		msg.getBody().add("foo", value);
		msg.getBody().add("bar", value);
		
		String[] names = msg.getBody().getNames();
		
		assertNotNull(names);
		
		assertEquals(names.length, 2);
		
		/*
		 * The array is not ordered.
		 */
		assertTrue("Check for bar", "bar".equals(names[0]) || "bar".equals(names[1]));
		assertTrue("Check for foo", "foo".equals(names[0]) || "foo".equals(names[1]));
		
		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
			
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			ExampleObject foo = (ExampleObject) nImpl.getBody().get("foo");
			
			assertEquals((foo.getValue() == value.getValue()), true);
			
			names = nImpl.getBody().getNames();
			
			assertNotNull(names);
			
			assertEquals(names.length, 2);
			/*
			 * The array is not ordered.
			 */
			assertTrue("Check for bar", "bar".equals(names[0]) || "bar".equals(names[1]));
			assertTrue("Check for foo", "foo".equals(names[0]) || "foo".equals(names[1]));
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testAddInvalidObject ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);

		try
		{
			msg.getBody().add("foo", new Object());
			
			fail();
		}
		catch (IllegalArgumentException ex)
		{
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}
	
	public void testRemoveObjects ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JAVA_SERIALIZED);

		assertEquals((msg != null), true);
		
		ExampleObject value = new ExampleObject(1234);
		
		msg.getBody().add("bar", value);
		
		msg.getBody().remove("bar");
		
		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
			
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			ExampleObject foo = (ExampleObject) nImpl.getBody().get("bar");
			
			assertEquals((foo == null), true);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}	
	}
	
	public void testJmsEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		JMSEpr epr = new JMSEpr(JMSEpr.TOPIC_TYPE, "foo", "bar");
		
		msg.getHeader().getCall().setTo(epr);
		
		try
		{
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof JMSEpr, true);
			
			assertEquals(((JMSEpr) theEpr).getConnectionFactory(), "bar");
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
	
	public void testHttpEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

		try
		{
			HTTPEpr epr = new HTTPEpr("http://www.foo.bar");
			
			msg.getHeader().getCall().setTo(epr);
			
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof HTTPEpr, true);
			
			assertEquals(theEpr.getURI().toString(), "http://www.foo.bar");
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
	
	public void testEmailEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

		try
		{
			EmailEpr epr = new EmailEpr(EmailEpr.SMTP_PROTOCOL, "foo.bar", "25", "me", "password");
			
			msg.getHeader().getCall().setTo(epr);
			
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof EmailEpr, true);
			
			assertEquals(((EmailEpr) theEpr).getPassword(), "password");
		}
		catch (Exception ex)
		{
			log.error(ex);
			ex.printStackTrace();
			
			fail(ex.toString());
		}
	}
	
	public void testFtpEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

		try
		{
			FTPEpr epr = new FTPEpr("ftp://www.foo.bar");
			
			epr.setPassive(true);
			
			msg.getHeader().getCall().setTo(epr);
			
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof FTPEpr, true);
			
			assertEquals(((FTPEpr) theEpr).getPassive(), true);
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
        
        public void testFtpsEPRType ()
        {
                Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

                try
                {
                        FTPSEpr epr = new FTPSEpr(new URI("ftps://www.foo.bar"));
                        
                        epr.setPassive(true);
                        
                        msg.getHeader().getCall().setTo(epr);
                        
                        ByteArrayOutputStream s = new ByteArrayOutputStream();
                        ObjectOutputStream o = new ObjectOutputStream(s);
        
                        o.writeObject(msg);
                        o.close();
                
                        ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
                        ObjectInputStream io = new ObjectInputStream(is);

                        MessageImpl nImpl = (MessageImpl) io.readObject();
                        
                        o.close();
                        
                        EPR theEpr = nImpl.getHeader().getCall().getTo();

                        assertEquals(theEpr instanceof FTPSEpr, true);
                }
                catch (Exception ex)
                {
                        log.error(ex);
                        
                        fail(ex.toString());
                }
        }
        
        public void testSftpEPRType ()
        {
                Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

                try
                {
                        SFTPEpr epr = new SFTPEpr(new URI("sftp://www.foo.bar"));
                        
                        epr.setPassive(true);
                        
                        msg.getHeader().getCall().setTo(epr);
                        
                        ByteArrayOutputStream s = new ByteArrayOutputStream();
                        ObjectOutputStream o = new ObjectOutputStream(s);
        
                        o.writeObject(msg);
                        o.close();
                
                        ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
                        ObjectInputStream io = new ObjectInputStream(is);

                        MessageImpl nImpl = (MessageImpl) io.readObject();
                        
                        o.close();
                        
                        EPR theEpr = nImpl.getHeader().getCall().getTo();

                        assertEquals(theEpr instanceof SFTPEpr, true);
                }
                catch (Exception ex)
                {
                        log.error(ex);
                        
                        fail(ex.toString());
                }
        }
	
	public void testJdbcEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

		try
		{
			JDBCEpr epr = new JDBCEpr("http://www.foo.bar", "SOME FAKE SQL");
			
			msg.getHeader().getCall().setTo(epr);
			
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof JDBCEpr, true);
			
			assertEquals(((JDBCEpr) theEpr).getSQL(), "SOME FAKE SQL");
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
	
	public void testFileEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

		try
		{
			FileEpr epr = new FileEpr("file://tmp/bar.txt");
			
			epr.setErrorDelete(true);
			
			msg.getHeader().getCall().setTo(epr);
			
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof FileEpr, true);
			
			assertEquals(((FileEpr) theEpr).getErrorDelete(), true);
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
	
	public void testSFtpEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);

		try
		{
			SFTPEpr epr = new SFTPEpr(new URI("http://www.foo.bar"), new URI("http://www.bar.foo"));

			assertEquals(epr.getCertificateURI().toString(), "http://www.bar.foo");
			
			msg.getHeader().getCall().setTo(epr);
			
			ByteArrayOutputStream s = new ByteArrayOutputStream();
			ObjectOutputStream o = new ObjectOutputStream(s);
	
			o.writeObject(msg);
			o.close();
		
			ByteArrayInputStream is = new ByteArrayInputStream(s.toByteArray());
			ObjectInputStream io = new ObjectInputStream(is);

			MessageImpl nImpl = (MessageImpl) io.readObject();
			
			o.close();
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();

			assertEquals(theEpr instanceof SFTPEpr, true);
			
			assertEquals(((SFTPEpr) theEpr).getCertificateURI().toString(), "http://www.bar.foo");
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
	
}
