/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import java.io.File;
import java.io.Serializable;

import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.annotation.AttachmentParam;
import org.jboss.soa.esb.actions.annotation.OnException;
import org.jboss.soa.esb.actions.annotation.OnSuccess;
import org.jboss.soa.esb.actions.annotation.Process;
import org.jboss.soa.esb.configure.ConfigProperty;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.annotation.Destroy;
import org.jboss.soa.esb.lifecycle.annotation.Initialize;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

import junit.framework.TestCase;

/**
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class BeanContainerActionUnitTest extends TestCase {
	
	@Override
	protected void setUp() throws Exception {
        ActionWithLifecycleWithProperty.throwInitException = false;
        ActionWithLifecycleWithProperty.throwProcessException = false;
	}

	public void test_isActionBean() {
		assertFalse(BeanContainerAction.isAnnotatedActionClass(EmptyBean.class));
		assertTrue(BeanContainerAction.isAnnotatedActionClass(ActionWithoutLifecycle.class));
	}

	public void test_ActionNoLifecycle() throws ConfigurationException, ActionProcessingException, ActionLifecycleException {
		ConfigTree configTree = new ConfigTree("conf");
		ActionWithoutLifecycleWithProperty action = new ActionWithoutLifecycleWithProperty();
		
		configTree.setAttribute("targetFile", "./somefile.xml");
		
		BeanContainerAction containerAction = null;
		
		try {
			containerAction = new BeanContainerAction(action, configTree);
			fail("Expected ConfigurationException");
		} catch(ConfigurationException e) {
			assertTrue(e.getMessage().startsWith("Property 'intDecodedProperty' not specified on configuration:"));
		}

		configTree.setAttribute("intDecodedProperty", "9876");		
		containerAction = new BeanContainerAction(action, configTree);
		
		Message message = MessageFactory.getInstance().getMessage();
		
		containerAction.initialise();
		containerAction.process(message);
		containerAction.processSuccess(message);
		containerAction.processException(message, new Exception());
		containerAction.destroy();
		
		assertNotNull(action.targetFile);
		assertEquals(action.targetFile, message.getBody().get());
	}

	public void test_ActionWithLifecycle_no_exception() throws ConfigurationException, ActionProcessingException, ActionLifecycleException {
        ConfigTree pipelineConfig = new ConfigTree("pipline");

        ConfigTree actionConfigTree = ActionProcessingPipelineUnitTest.addAction(pipelineConfig, ActionWithLifecycleWithProperty.class.getName());

        ActionProcessingPipeline pipeline = new ActionProcessingPipeline(pipelineConfig);
        pipeline.initialise();
		
        try {
			Message message = MessageFactory.getInstance().getMessage();
			pipeline.process(message);
        } finally {
        	pipeline.destroy();
        }
		
		assertTrue(ActionWithLifecycleWithProperty.init1Called);
		assertTrue(ActionWithLifecycleWithProperty.init2Called);
		assertEquals(actionConfigTree, ActionWithLifecycleWithProperty.config);
		assertTrue(ActionWithLifecycleWithProperty.processCalled);
		assertTrue(ActionWithLifecycleWithProperty.onsuccess1Called);
		assertTrue(ActionWithLifecycleWithProperty.onsuccess2Called);
		assertFalse(ActionWithLifecycleWithProperty.onexception1Called);
		assertFalse(ActionWithLifecycleWithProperty.onexception2Called);
		assertFalse(ActionWithLifecycleWithProperty.onexception3Called);
		assertTrue(ActionWithLifecycleWithProperty.destroyCalled);
	}

	public void test_ActionWithLifecycle_init_exception() throws ConfigurationException, ActionProcessingException, ActionLifecycleException {
        ConfigTree pipelineConfig = new ConfigTree("pipline");

        ActionProcessingPipelineUnitTest.addAction(pipelineConfig, ActionWithLifecycleWithProperty.class.getName());
        ActionWithLifecycleWithProperty.throwInitException = true;

        ActionProcessingPipeline pipeline = new ActionProcessingPipeline(pipelineConfig);
        try {
        	pipeline.initialise();
        	fail("Expected ConfigurationException");
        } catch(ConfigurationException e) {
        	assertEquals("Unexpected exception during lifecycle initialisation", e.getMessage());
        	assertEquals("init exception", e.getCause().getMessage());
        }
	}

	public void test_ActionWithLifecycle_process_exception() throws ConfigurationException, ActionProcessingException, ActionLifecycleException {
        ConfigTree pipelineConfig = new ConfigTree("pipline");

        ConfigTree actionConfigTree = ActionProcessingPipelineUnitTest.addAction(pipelineConfig, ActionWithLifecycleWithProperty.class.getName());

        ActionProcessingPipeline pipeline = new ActionProcessingPipeline(pipelineConfig);
        ActionWithLifecycleWithProperty.throwProcessException = true;
        pipeline.initialise();
		
        try {
			Message message = MessageFactory.getInstance().getMessage();
			pipeline.process(message);
        } finally {
        	pipeline.destroy();
        }
		
		assertTrue(ActionWithLifecycleWithProperty.init1Called);
		assertTrue(ActionWithLifecycleWithProperty.init2Called);
		assertEquals(actionConfigTree, ActionWithLifecycleWithProperty.config);
		assertTrue(ActionWithLifecycleWithProperty.processCalled);
		assertFalse(ActionWithLifecycleWithProperty.onsuccess1Called);
		assertFalse(ActionWithLifecycleWithProperty.onsuccess2Called);
		assertTrue(ActionWithLifecycleWithProperty.onexception1Called);
		assertTrue(ActionWithLifecycleWithProperty.onexception2Called);
		assertTrue(ActionWithLifecycleWithProperty.onexception3Called);
		assertTrue(ActionWithLifecycleWithProperty.destroyCalled);
	}
	
	public void test_paramResolution_01() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		ActionA action = new ActionA();
		
		configTree.setAttribute("targetFile", "./somefile.xml");
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add(new PayloadObj("aPay"));
		containerAction.process(message);
		
		assertEquals("aPay", message.getBody().get());		
	}
	
	public void test_paramResolution_02() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		ActionA action = new ActionA();
		
		configTree.setAttribute("targetFile", "./somefile.xml");
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("Not the action arg");
		message.getBody().add("someother location", new PayloadObj("aPay"));
		containerAction.process(message);
		
		assertEquals("aPay", message.getBody().get());		
	}
	
	public void test_paramResolution_03() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		ActionA action = new ActionA();
		
		configTree.setAttribute("targetFile", "./somefile.xml");
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("Not the action arg");
		message.getProperties().setProperty("payX", new PayloadObj("aPay"));
		containerAction.process(message);
		
		assertEquals("aPay", message.getBody().get());		
	}
	
	public void test_paramResolution_04() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		ActionA action = new ActionA();
		
		configTree.setAttribute("targetFile", "./somefile.xml");
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("Not the action arg");
		message.getAttachment().put("payX", new PayloadObj("aPay"));
		
		containerAction.process(message);
		
		assertEquals("aPay", message.getBody().get());		
	}
	
	public void test_paramResolution_05() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		ActionB action = new ActionB();
		
		configTree.setAttribute("targetFile", "./somefile.xml");
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("BodyString");
		message.getAttachment().put("payX", new PayloadObj("aPay"));
		containerAction.process(message);
		
		assertEquals("BodyString-aPay", message.getBody().get());		
	}
	
	public void test_JBESB_3547() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		RuntimeExceptionThrowerAction action = new RuntimeExceptionThrowerAction();
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("BodyString");
		try {
			containerAction.process(message);
		} catch(RuntimeException e) {
			assertEquals(RuntimeExceptionThrowerAction.EXCEPTION_MESSAGE, e.getMessage());
		}
	}
	
	public void test_JBESB_3548_null_return() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		NullReturnAction action = new NullReturnAction();
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("BodyString");
		assertNull(containerAction.process(message));
	}
	
	public void test_JBESB_3548_void_return() throws ConfigurationException, ActionProcessingException {
		ConfigTree configTree = new ConfigTree("conf");
		VoidReturnAction action = new VoidReturnAction();
		
		BeanContainerAction containerAction = new BeanContainerAction(action, configTree);
		Message message = MessageFactory.getInstance().getMessage();
		
		message.getBody().add("BodyString");
		assertTrue(message == containerAction.process(message));
	}
	
	public class EmptyBean {		
	}
	
	public class ActionWithoutLifecycle {
		@Process
		public void processMethod(Message m) {			
		}
	}
	
	public static class ActionWithoutLifecycleWithProperty {
		@ConfigProperty
		private File targetFile;
		@ConfigProperty
		private int intDecodedProperty;
		
		@Process
		public void processMethod(Message m) {
			m.getBody().add(targetFile);
		}		
	}
	
	public static class ActionWithLifecycleWithProperty {
		
		private static boolean throwInitException;		
		private static boolean throwProcessException;
		private static boolean init1Called;
		private static boolean init2Called;
		private static boolean processCalled;
		private static boolean destroyCalled;
		private static ConfigTree config;
		private static boolean onsuccess1Called;
		private static boolean onsuccess2Called;
		private static boolean onexception1Called;
		private static boolean onexception2Called;
		private static boolean onexception3Called;		
		
		public ActionWithLifecycleWithProperty() {
			init1Called = false;
			init2Called = false;
			processCalled = false;
			destroyCalled = false;
			config = null;
			onsuccess1Called = false;
			onsuccess2Called = false;
			onexception1Called = false;
			onexception2Called = false;
			onexception3Called = false;		
		}
		
		@Initialize
		public void init1() {
			init1Called = true;
		}				
		
		@Initialize
		public void init2(ConfigTree config) throws ActionLifecycleException {
			this.config = config;
			init2Called = true;
			if(throwInitException) {
				throw new ActionLifecycleException("init exception");
			}
		}				
		
		@Process
		public void process(Message m) throws ActionProcessingException {
			processCalled = true;
			if(throwProcessException) {
				throw new ActionProcessingException("process exception");
			}			
		}		
		
		@OnSuccess
		public void onSuccess1() {
			onsuccess1Called = true;
		}
		
		@OnSuccess
		public void onSuccess2(Message message) {
			assertNotNull("message passed to onSuccess method is null", message);
			onsuccess2Called = true;
		}
		
		@OnException
		public void onException1() {
			onexception1Called = true;
		}
		
		@OnException
		public void onException2(Message message) {
			assertNotNull("message passed to onException method is null", message);
			onexception2Called = true;
		}
		
		@OnException
		public void onException3(Message message, Throwable th) {
			assertNotNull("message passed to onException method is null", message);
			assertNotNull("throwable passed to onException method is null", th);
			assertTrue(th instanceof ActionProcessingException);
			assertEquals("process exception", th.getMessage());
			onexception3Called = true;
		}
		
		@Destroy
		public void destroy() {
			destroyCalled = true;
		}		
	}
	
	public class ActionA {
		@Process
		public String processA(PayloadObj pay) {
			return pay.toString();
		}
	}
	
	public class ActionB {
		@Process
		public String processB(String bodyString, @AttachmentParam PayloadObj pay) {
			return bodyString + "-" + pay.toString();
		}
	}
	
	public class RuntimeExceptionThrowerAction {
		
		private static final String EXCEPTION_MESSAGE = "Some runtime error happened....";
		
		@Process
		public String process(String bodyString) {
			throw new RuntimeException(EXCEPTION_MESSAGE);
		}
	}	
	
	public class NullReturnAction {
		
		@Process
		public String process(String bodyString) {
			return null;
		}
	}	
	
	public class VoidReturnAction {
		
		@Process
		public void process(String bodyString) {
		}
	}	
	
	public class PayloadObj implements Serializable {
		
		private String name;
		
		public PayloadObj(String name) {
			this.name = name;
		}
		
		@Override
		public String toString() {
			return name;
		}		
	}
}
