/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.message;

import java.io.Serializable;
import java.net.URL;
import java.util.concurrent.TimeUnit;

import javax.crypto.SealedObject;
import javax.security.auth.Subject;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.security.PublicCryptoUtil;
import org.jboss.soa.esb.services.security.SecurityConfig;
import org.jboss.soa.esb.services.security.SecurityConfigUtil;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.TestPrincipal;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.principals.User;
import org.jboss.soa.esb.util.ClassUtil;

public class ActionProcessingPipelineUnitTest extends TestCase
{
    private static final String DOMAIN = "SuccessfulLogin" ;
    private static final String DIFF_DOMAIN = "UnsuccessfulLogin" ;
    
	private String jbossEsbProperties;

    public void testProperty() {

	}

    @Override
    protected void setUp() throws Exception
    {
        super.setUp();
        MockActionInfo.reset();
        System.setProperty("javax.xml.registry.ConnectionFactoryClass","org.apache.ws.scout.registry.ConnectionFactoryImpl");

        MockRegistry.install();
        LifecycleResourceManager.getSingleton().associateDeployment(getClass().getCanonicalName()) ;

        jbossEsbProperties = System.getProperty(Environment.PROPERTIES_FILE);
        URL resource = ClassUtil.getResource("security-properties.xml", getClass());
        System.setProperty(Environment.PROPERTIES_FILE, "abs://" + resource.getFile());

        URL loginConfig = ClassUtil.getResource("test_jaas.config", getClass());
        System.setProperty("java.security.auth.login.config", loginConfig.getFile());
    }

    @Override
    protected void tearDown() throws Exception
    {
        LifecycleResourceManager.getSingleton().disassociateDeployment(getClass().getCanonicalName()) ;
        if ( jbossEsbProperties != null )
            System.setProperty(Environment.PROPERTIES_FILE, jbossEsbProperties);
        super.tearDown();
    }

    /*
     * Tests to run
     *  - create a pipeline with a failure, make sure failure is called in correct order and subsequent actions are not called.
     *    Check initialise and destroy called.
     *
     *  Do the above for each type of action.
     */
    public void testActionPipelineProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList(), 4, 2, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList(), 4, 2, 0, 4, 2, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 4, 2, 0) ;
    }

    public void testActionPipelineProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessorFailure.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 2, 0) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 0, 2) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 2, 0, 2, 0) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 4, 2, 0) ;
    }

    public void testOverriddenActionPipelineProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 4, 3, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7, 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 4, 3, 0, 7, 4, 3, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }

    public void testOverriddenActionPipelineProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionPipelineProcessorFailure.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 2, 0) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 2, 0, 5, 2, 0) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }

    public void testOverriddenActionLifecycleProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            null, "processSuccess", null) ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
                null, null, "processException") ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7, 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2, 7, 2) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }

    public void testOverriddenActionLifecycleProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionLifecycleProcessorFailure.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 5) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }

    public void testOverriddenActionProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionProcessor.class.getName(),
            null, "processSuccess", null) ;
        addAction(configTree, MockActionProcessor.class.getName(),
                null, null, "processException") ;
        addAction(configTree, MockActionProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7, 8, 10, 12, 14, 15) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2, 15, 10) ;
        checkOrder(MockActionInfo.getExceptionList()) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testOverriddenActionProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionProcessorFailure.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;

        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 8, 10) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 11) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testErrorActionProcessorException() throws Exception
	{
		final ConfigTree configTree = new ConfigTree("parent");
		configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
				ListenerTagNames.MEP_ONE_WAY) ;

		addAction(configTree, ErrorActionProcessor.class.getName(), "process",
				null, null);
		addAction(configTree, ErrorActionProcessor.class.getName(), null,
				"anotherProcessSuccess", null);
		addAction(configTree, MockActionProcessorFailure.class.getName(), null,
				null, "anotherProcessException");
		addAction(configTree, ErrorActionProcessor.class.getName(),
				"process, anotherProcess", "anotherProcessSuccess",
				"anotherProcessException");

		final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(
				configTree);
		pipeline.initialise();

		final boolean result = pipeline.process(MessageFactory.getInstance()
				.getMessage());

		assertFalse(result);
	}

    public void testNPEActionProcessorException() throws Exception
	{
		final ConfigTree configTree = new ConfigTree("parent");
		configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
				ListenerTagNames.MEP_ONE_WAY) ;

		addAction(configTree, NPEActionProcessor.class.getName(), "process",
				null, null);
		addAction(configTree, NPEActionProcessor.class.getName(), null,
				"anotherProcessSuccess", null);
		addAction(configTree, MockActionProcessorFailure.class.getName(), null,
				null, "anotherProcessException");
		addAction(configTree, ErrorActionProcessor.class.getName(),
				"process, anotherProcess", "anotherProcessSuccess",
				"anotherProcessException");

		final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(
				configTree);
		pipeline.initialise();

		final boolean result = pipeline.process(MessageFactory.getInstance()
				.getMessage());

		assertFalse(result);
	}

    public void testInvalidConstructor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionPipelineProcessorConstructorFailure.class.getName(),
            null, null, null) ;
        try
        {
            new ActionProcessingPipeline(configTree) ;
            fail("Expected ConfigurationException") ;
        }
        catch (final ConfigurationException ce)
        {

        }
    }

    public void testSecuredWithoutExistingSecurityContextOrAuthenticationRequest() throws ConfigurationException
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        addSecurityConfig(configTree, "adminRole", null, DOMAIN, null);
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;
        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree);
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;

        final boolean processingResult = pipeline.process(MessageFactory.getInstance().getMessage());
        assertFalse("The Service was configured with security but neither a AuthenticationContext nor a SecurityContext is made available", processingResult);

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testSecuredAuthentication() throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        // Add the security configuration.
        addSecurityConfig(configTree, "adminRole", null, DOMAIN, null);
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;

        final Message message = MessageFactory.getInstance().getMessage();

        //  Create an AuthenticationRequest which is needed to authenticate if the security
        final AuthenticationRequest authRequest = new AuthenticationRequestImpl.Builder().build();
        final byte[] encryptedAuthRequest = PublicCryptoUtil.INSTANCE.encrypt((Serializable) authRequest);
        message.getContext().setContext(SecurityService.AUTH_REQUEST, encryptedAuthRequest);

        final boolean processingResult = pipeline.process(message);

        assertTrue(processingResult);
        assertEquals(new TestPrincipal("test").getName(), MockSecuredActionProcessor.getSubject().getPrincipals().iterator().next().getName());
        assertEquals("The encrypted AuthenticationRequest should have been set as a thread local",
                encryptedAuthRequest, AuthenticationRequestImpl.getEncryptedAuthRequest());
        assertTrue("The SecurityContext should have been pushed", MockSecurityContextPropagator.wasPushCalled());
        assertTrue("The SerirityContext should have been popped", MockSecurityContextPropagator.wasPopCalled());
        assertNull("AuthenticationRequest should have been removed from the message context",
                message.getContext().getContext(SecurityService.AUTH_REQUEST));

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testSecuredWithPreExistingSecurityContext() throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        addSecurityConfig(configTree, "adminRole", null, DOMAIN, null);
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;

        final Subject subject = new Subject();
        final User user = new User("AustinPowerwich");
        final byte[] publicCred = "publicsecret".getBytes();
        subject.getPrincipals().add(user);
        subject.getPublicCredentials().add(publicCred);

        //  Create and encrypt the security context. This simulates a call for a service that has already been authentcated.
        final SecurityContext securityContext = new SecurityContext(subject, SecurityContext.getConfigurationTimeout(), DOMAIN);
        final SealedObject sealedObject = SecurityContext.encryptContext(securityContext);
        final Message message = MessageFactory.getInstance().getMessage();
        message.getContext().setContext(SecurityService.CONTEXT, sealedObject);

        final boolean processingResult = pipeline.process(message);
        assertTrue(processingResult);
        assertEquals(user, MockSecuredActionProcessor.getSubject().getPrincipals().iterator().next());

        assertNull(message.getContext().getContext(SecurityService.CONTEXT));

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testSecuredWithExistingSecurityContextWhichHasExpired() throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        addSecurityConfig(configTree, "adminRole", null, DOMAIN, null);
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;


        Subject subject = new Subject();
        //  add principal
        User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        //  add public credentials
        byte[] publicCred = "publicsecret".getBytes();
        subject.getPublicCredentials().add(publicCred);

        //  Create and encrypt the security context. This simulates a call for a service
        //  that has already been authentcated...but with a very short timeout.
        final SecurityContext securityContext = new SecurityContext(subject, 10, DOMAIN);

        TimeUnit.SECONDS.sleep(1);

        final SealedObject sealedObject = SecurityContext.encryptContext(securityContext);
        final Message message = MessageFactory.getInstance().getMessage();
        message.getContext().setContext(SecurityService.CONTEXT, sealedObject);

        final boolean processingResult = pipeline.process(message);
        assertFalse("Processing should have failed as the SecurityContext was invalid(expired)", processingResult);

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testSecuredWithSecurityContextOverriddenTimeout() throws Exception
    {
        final long timeout = 90000;

        //  setup config tree
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;
        final ConfigTree securityConfigTree = addSecurityConfig(configTree, "adminRole", null, DOMAIN, null, timeout);
        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        final SecurityConfig securityConfig = SecurityConfigUtil.createSecurityConfig(securityConfigTree);

        final long contextTimeout = pipeline.getSecurityContextTimeout(securityConfig);

        assertEquals("Timeout should be equals to the one in securty element(jboss-esb.xml) and not the global timeout(jbossesb-properties.xml",
                timeout, contextTimeout);

        pipeline.destroy() ;
    }

    public void testSecuredWithSecurityContextDefaultTimeout() throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;
        final ConfigTree securityConfigTree = addSecurityConfig(configTree, "adminRole", null, DOMAIN, null);
        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;

        final SecurityConfig securityConfig = SecurityConfigUtil.createSecurityConfig(securityConfigTree);
        final long contextTimeout = pipeline.getSecurityContextTimeout(securityConfig);

        assertEquals(30000, contextTimeout);

        pipeline.destroy() ;
    }

    public void testSecuredWithSecurityContextAndInvalidRoles() throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;
        addSecurityConfig(configTree, "adminRole", null, DOMAIN, "noRole");

        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;

        final Subject subject = new Subject();
        final User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        final byte[] publicCred = "publicsecret".getBytes();
        subject.getPublicCredentials().add(publicCred);

        final SecurityContext securityContext = new SecurityContext(subject, SecurityContext.getConfigurationTimeout(), DOMAIN);
        final SealedObject sealedObject = SecurityContext.encryptContext(securityContext);
        final Message message = MessageFactory.getInstance().getMessage();
        message.getContext().setContext(SecurityService.CONTEXT, sealedObject);

        final boolean processingResult = pipeline.process(message);
        assertFalse("Processing should have failed as the caller was not in any of the allowed roles", processingResult);

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public void testSecuredWithDifferentDomain() throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        addSecurityConfig(configTree, "adminRole", null, DOMAIN, null);
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockSecuredActionProcessor.class.getName(), "process", null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;


        Subject subject = new Subject();
        //  add principal
        User user = new User("AustinPowerwich");
        subject.getPrincipals().add(user);
        //  add public credentials
        byte[] publicCred = "publicsecret".getBytes();
        subject.getPublicCredentials().add(publicCred);

        //  Create and encrypt the security context. This simulates a call for a service
        //  that has already been authenticated...but with a very long timeout.
        final SecurityContext securityContext = new SecurityContext(subject, 1000 * 60 * 60 * 24 * 7, DIFF_DOMAIN);

        final SealedObject sealedObject = SecurityContext.encryptContext(securityContext);
        final Message message = MessageFactory.getInstance().getMessage();
        message.getContext().setContext(SecurityService.CONTEXT, sealedObject);

        final boolean processingResult = pipeline.process(message);
        assertFalse("Processing should have failed as the SecurityContext was for a different domain", processingResult);

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }

    public static ConfigTree addAction(final ConfigTree configTree, final String actionName) {
        return addAction(configTree, actionName, null, null, null);
    }

    public static ConfigTree addAction(final ConfigTree configTree, final String actionName,
        final String processOverride, final String successOverride,
        final String exceptionOverride)
    {
        final ConfigTree actionChild = new ConfigTree(ListenerTagNames.ACTION_ELEMENT_TAG, configTree) ;
        actionChild.setAttribute(ListenerTagNames.ACTION_CLASS_TAG, actionName) ;
        if (processOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.PROCESS_METHOD_TAG, processOverride) ;
        }
        if (successOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.NORMAL_COMPLETION_METHOD_TAG, successOverride) ;
        }
        if (exceptionOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.EXCEPTION_METHOD_TAG, exceptionOverride) ;
        }
        
        return actionChild;
    }

    private void checkOrder(final Integer[] list, int ... values)
    {
        final int numValues = (values == null ? 0 : values.length) ;
        final int listLength = (list == null ? 0 : list.length) ;

        assertEquals("Unexpected list/values count", numValues, listLength) ;

        for(int count = 0 ; count < numValues ; count++)
        {
            assertEquals("Unexpected call order at count " + count, values[count], list[count].intValue()) ;
        }
    }

    private ConfigTree addSecurityConfig(
            final ConfigTree parent,
            final String runAs,
            final String callerIdentity,
            final String moduleName,
            final String rolesAllowed)
    {
        final ConfigTree securityElement = new ConfigTree(ListenerTagNames.SECURITY_TAG, parent);
        securityElement.setAttribute(ListenerTagNames.RUN_AS_TAG, runAs);
        securityElement.setAttribute(ListenerTagNames.USE_CALLERS_IDENTIDY_TAG, callerIdentity);
        securityElement.setAttribute(ListenerTagNames.MODULE_NAME_TAG, moduleName);
        if ( rolesAllowed != null )
        {
            securityElement.setAttribute(ListenerTagNames.ROLES_ALLOWED, rolesAllowed);
        }
        return securityElement;
    }

    private ConfigTree addSecurityConfig(
            final ConfigTree parent,
            final String runAs,
            final String callerIdentity,
            final String moduleName,
            final String rolesAllowed,
            final long timeout)
    {
        ConfigTree securityElement = this.addSecurityConfig(parent, runAs, callerIdentity, moduleName, rolesAllowed);
        if (timeout != 0l)
        {
            ConfigTree property = new ConfigTree("property", securityElement);
            property.setAttribute("name", Environment.SECURITY_SERVICE_CONTEXT_TIMEOUT);
            property.setAttribute("value", Long.toString(timeout));
        }
        return securityElement;
    }

}
