/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;


import javax.jms.JMSException;
import javax.jms.Queue;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;
import org.jboss.soa.esb.testutils.SerializableMockQueue;
import org.junit.Before;
import org.junit.Test;
import org.mockejb.jms.MockQueue;
import org.mockejb.jms.TextMessageImpl;

/**
 * Unit test for DefaultESBPropertiesSetter
 * </p>
 *
 * @author <a href="mailto:daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @since 4.2
 */
public class DefaultESBPropertiesSetterUnitTest
{
	@SuppressWarnings ("unused")
	private Logger log = Logger .getLogger( DefaultESBPropertiesSetterUnitTest.class );

	private final static String jmsCorrelationID = "YYXX-123456780-GG";
	private ESBPropertiesSetter strategy;
	private javax.jms.Message fromJMSTextMessage;
	private Message toESBMessage;

	@Test
	public void setPropertiesFromJMSMessage_CorrelationID() throws JMSException
	{
		fromJMSTextMessage.setJMSCorrelationID( jmsCorrelationID );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );

		assertEquals( jmsCorrelationID, toESBMessage.getHeader().getCall().getRelatesTo().getFragment() );
	}

	@Test
	public void setPropertiesFromJMSMessageReplyTo() throws JMSException
	{
		final String destinationName = "/queue/testDest";
		final MockQueue queue = new SerializableMockQueue( destinationName );
		fromJMSTextMessage.setJMSReplyTo( queue );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );

		final Object object = toESBMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_REPLY_TO );
		assertNotNull( object );
		assertTrue( object instanceof Queue );
	}
	
	@Test
	public void setPropertiesFromJMSMessageReplyToNull() throws JMSException
	{
		final MockQueue queue = null;
		fromJMSTextMessage.setJMSReplyTo( queue );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );

		final String replyTo = (String) toESBMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_REPLY_TO );
		assertNull( replyTo );
	}

	@Test
	public void setPropertiesFromJMSMessage_Expiration() throws JMSException, InterruptedException
	{
		final long ttl = 5000l;
		final long expirationTime = System.currentTimeMillis() + ttl;
		fromJMSTextMessage.setJMSExpiration( expirationTime );

		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );

		Long actualExpiration = (Long)toESBMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_EXPIRATION );
		assertTrue( actualExpiration != 0 );
		assertTrue( actualExpiration > 0 );
	}

	@Test
	public void setPropertiesFromJMSMessage_withProperties() throws JMSException
	{
		final String propertyKey = "myProperty";
		final String propertyValue = "myPropertyValue";
		fromJMSTextMessage.setStringProperty( propertyKey, propertyValue );

		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );

		assertEquals( propertyValue, toESBMessage.getProperties().getProperty( propertyKey )) ;
	}

	@Test
	public void setPropertiesFromJMSMessage_withJMSRedelivererdProperty_false() throws JMSException
	{
		boolean redelivered = false;
		fromJMSTextMessage.setJMSRedelivered( redelivered );
		
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		assertEquals( redelivered, toESBMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_REDELIVERED )) ;
	}
	
	@Test
	public void setPropertiesFilterAllVendorSpecificProperties() throws JMSException
	{
	    final String filter = "^(JMS_).*";
	    DefaultESBPropertiesSetter setter = new DefaultESBPropertiesSetter(filter);
	    
		final String propertyKey = "JMS_SomeVendor_Property";
		final String propertyValue = "myPropertyValue";
		fromJMSTextMessage.setStringProperty( propertyKey, propertyValue );
		
		setter.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		Properties properties = toESBMessage.getProperties();
		assertNull(properties.getProperty(propertyKey));
	}
	
	@Test
    public void setPropertiesFilterIBMQVendorSpecificProperties() throws JMSException
    {
        final String filter = "^(JMS_IBMQ_).+";
        DefaultESBPropertiesSetter setter = new DefaultESBPropertiesSetter(filter);
        
        final String propertyKey = "JMS_IBMQ_Property";
        final String propertyValue = "myPropertyValue";
        fromJMSTextMessage.setStringProperty( propertyKey, propertyValue );
        
        setter.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
        
        Properties properties = toESBMessage.getProperties();
		assertNull(properties.getProperty(propertyKey));
    }
	
	@Before
	public void setup()
	{
		fromJMSTextMessage = new TextMessageImpl();
		toESBMessage = MessageFactory.getInstance().getMessage();
		strategy = new DefaultESBPropertiesSetter();
	}

	/*
	 * Just here to help Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( DefaultESBPropertiesSetterUnitTest.class );
	}

}
