/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.helpers;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.testutils.StringUtils;
import org.jboss.soa.esb.common.tests.BaseTest;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Unit tests for the ConfigTree class.
 * @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 */
public class ConfigTreeUnitTest extends BaseTest {
	private Logger log = Logger.getLogger( ConfigTreeUnitTest.class );

	public void test_fromInputStream_args() throws SAXException, IOException {
		try {
			ConfigTree.fromInputStream(null);
			fail("Expected IllegalArgumentException on null stream.");
		} catch(IllegalArgumentException e) {
			// Expected
		}
	}
	
	private InputStream getStream(String sName)
	{
		InputStream oRet =getClass().getResourceAsStream(sName);
		if (null==oRet)
			try { oRet = new FileInputStream(sName); }
			catch(IOException e) { /* OK  Just fall through */}
		return oRet;
	}
	
	public void test_fromInputStream_and_toXML() throws SAXException, IOException
	{
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile1.xml"));
		byte[] expected = StreamUtils.readStream(getStream("expected_01.xml"));
		
		// Create the ConfigTree from a Stream.  Then...
		// Dump the XML to a buffer and compare it with the expected - checking that the
		// Created ConfigTree DOM is the same as what was supplied in the stream.  Also
		// tests the toXML method.  If this fails, either fromInputStream is not working correctly
		// or the toXML method is not serialising correctly.
		String output = null;
		try { output = confTree.toXml(); }
		catch (Exception e) 
		{ 
			log.error(e);
		}
		assertTrue("ConfigTree creation from a stream failed, or, ConfigTree toXML failed - failed to produce the same XML.", 
				StringUtils.equalsIgnoreLinebreaks(new String(expected), output, false));
	}
        
        public void testChildren () throws SAXException, IOException
        {
                ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile5.xml"));
                
                System.err.println(confTree.toString());
                
                ConfigTree[] children = confTree.getAllChildren();
                
                for (ConfigTree child : children)
                {
                    System.err.println(child.toString()+"\n");
                }
                
                // make sure attribute names are unique!
                
                ConfigTree theChild = confTree.getFirstChild("message-property-name");
                
                System.err.println("\n"+theChild);
                
                assertNotNull(theChild);
        }

	public void test_fromXML_args() throws SAXException, IOException
	{		
		try {
			ConfigTree.fromXml(null);
			fail("Expected IllegalArgumentException on null String.");
		} catch(IllegalArgumentException e) {
			// Expected
		}
		try {
			ConfigTree.fromXml(" ");
			fail("Expected SAXParseException on null String.");
		} catch(SAXParseException e) {
			// Expected
		}
	}

	public void test_fromXML_and_toXML() throws SAXException, IOException
	{
		byte[] source = StreamUtils.readStream(getStream("expected_01.xml"));
		ConfigTree confTree = ConfigTree.fromXml(new String(source));
		byte[] expected = source;
		
		// Create the ConfigTree from a String.  Then...
		// Dump the XML to a buffer and compare it with the expected - checking that the
		// Created ConfigTree DOM is the same as what was supplied in the stream.  Also
		// tests the toXML method.  If this fails, either fromInputStream is not working correctly
		// or the toXML method is not serialising correctly.
		String output = confTree.toXml();
		assertTrue("ConfigTree creation from a String failed, or, ConfigTree toXML failed - failed to produce the same XML.", 
				StringUtils.equalsIgnoreLinebreaks(new String(expected), output, false));
	}

	public void test_Constructor_Clone() throws SAXException, IOException {
		ConfigTree conf1 = ConfigTree.fromInputStream(getStream("testfile1.xml"));
//		ByteArrayOutputStream output;
		byte[] expected = StreamUtils.readStream(getStream("expected_01.xml"));	
		// Create a new ConfigTree from an existing one and compare it with the expected.
		ConfigTree conf2 = conf1.cloneObj();
		assertTrue("ConfigTree creation from a stream failed, or, cloneObj failed - failed to produce the same XML.", 
				StringUtils.equalsIgnoreLinebreaks(new String(expected), conf2.toXml(), false));
	}

	public void test_Constructor_String_withoutparent() throws SAXException, IOException {
		// Create a new ConfigTree from without a parent ConfigTree.
		ConfigTree confTree = new ConfigTree("newConfigTree",null);
		
		assertEquals("New ConfigTree invalid.", "<newConfigTree/>" , confTree.toXml());
	}
	
	public void test_getName() {
		ConfigTree confTree = new ConfigTree("newConfigTree",null);
		assertEquals("getName returned invalid data.", "newConfigTree", confTree.getName());
	}
	
	public void test_getAttrKeys_hasnokeys() throws SAXException, IOException {
		ConfigTree confTree = new ConfigTree("newConfigTree",null);		
		assertTrue("invalid attribute name list", confTree.attributeCount()== 0);
	}
	
	public void test_getAttrKeys_haskeys() throws SAXException, IOException {
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile2.xml"));
		String[] expected = new String[] {"attrib1", "attrib2", "attrib3"};
		Arrays.sort(expected);
		String[] actual = confTree.getAttributeNames().toArray(new String[confTree.attributeCount()]);
		Arrays.sort(actual);
		
		assertTrue("invalid attribute name list", Arrays.equals(expected, actual));
	}
	
	public void test_getAttr() throws SAXException, IOException {
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile2.xml"));
		
		assertEquals("invalid attribute value", "value2", confTree.getAttribute("attrib2"));
		assertEquals("invalid attribute value", null, confTree.getAttribute("attribXX"));
	}
	
	public void test_getTextChildren() throws SAXException, IOException {
		// This test also tests the static method getTextValue. 
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile3.xml"));
		
		// REVIEW: These tests capture the current behavior of the getTextChildren method.  I'm not convinced this behavior is correct!!
		// I think perhaps this method should not be depending on getTextValue because getTextValue
		// concatentates the child text nodes.
		assertTrue("invalid Text Children list", 
				"Some root text... and some more root text...".equals(confTree.getWholeText()));
		assertTrue("invalid Text Children list", 
				Arrays.equals(new String[] {"", "Some nested text..."}, confTree.getTextChildren("el")));
	}

	public void test_getElementChildren() throws SAXException, IOException {
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile3.xml"));
		ConfigTree[] confTrees;

		String expected = new String(StreamUtils.readStream(getStream("expected_03.xml"))).trim();
		assertEquals("Wrong ConfigTree ElementChild value.", expected, confTree.toString().trim());

		// Test the nested <el> elements - of which there are 2...
		confTrees = confTree.getChildren("el");
		assertEquals("Wrong number of ConfigTree ElementChildren.", 2, confTrees.length);
		assertEquals("Wrong ConfigTree ElementChild value.", "<el index=\"1\"/>", confTrees[0].toString().trim());
		assertEquals("Wrong ConfigTree ElementChild value.", "<el index=\"2\">Some nested text...</el>", confTrees[1].toString().trim());

		// Test for a non-existant element...
		confTrees = confTree.getChildren("xxxx");
		assertEquals("Wrong number of ConfigTree ElementChildren.", 0, confTrees.length);
	}
	
	public void test_getFirstElementChild() throws SAXException, IOException {
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile3.xml"));
		
		String expected = new String(StreamUtils.readStream(getStream("expected_03.xml"))).trim();
		assertEquals("Wrong ConfigTree ElementChild value.", expected, confTree.toString().trim());

		// Test the nested <el> elements - of which there are 2...
		confTree = confTree.getFirstChild("el");
		assertEquals("Wrong ConfigTree ElementChild value.", "<el index=\"1\"/>", confTree.toString().trim());

		// Test for a non-existant element...
		confTree = confTree.getFirstChild("xxxx");
		assertEquals("Expected null ConfigTree ElementChildren for non-existant element.", null, confTree);
	}

	public void test_getAllElemChildren() throws SAXException, IOException {
		ConfigTree confTree = ConfigTree.fromInputStream(getStream("testfile3.xml"));
		ConfigTree[] confTrees;

		confTrees = confTree.getAllChildren();
		assertEquals("Wrong number of ConfigTree ElementChildren.", 2, confTrees.length);
		assertEquals("Wrong ConfigTree ElementChild value.", "<el index=\"1\"/>", confTrees[0].toString().trim());
		assertEquals("Wrong ConfigTree ElementChild value.", "<el index=\"2\">Some nested text...</el>", confTrees[1].toString().trim());

		// REVIEW: Note, this method only returns ConfigTree instances for the child elements
		// of the document root node. Comparing this to the behaviour of getFirstElementChild
		// would seem to demonstrate an inconsitency???  To be consistent, I would expect this method 
		// to return a list of 3 ConfigTree instances - the <root> and 2 <el> elements.
	}
	
	public void test_setAttr() throws SAXException, IOException {
		ConfigTree confTree = new ConfigTree("newConfigTree");
		
		confTree.setAttribute("attrib1", "value1");
		assertEquals("value1", confTree.getAttribute("attrib1"));

		// A null value removes the attribute.
		confTree.setAttribute("attrib1", null);
		assertEquals(null, confTree.getAttribute("attrib1"));

		try { confTree.setAttribute(null, null); }
		catch (IllegalArgumentException e) {/* OK  we expect this */}
		catch (Exception e2) {fail(e2.getMessage());}
	}
	
	public void test_addElemChild() throws SAXException, IOException {
		ConfigTree ConfigTree1 = new ConfigTree("newConfigTree");
		ConfigTree ConfigTreeX = new ConfigTree("X",ConfigTree1);
		new ConfigTree("Y1",ConfigTree1);
		new ConfigTree("Y2",ConfigTreeX);
		
		assertEquals("<newConfigTree><X><Y2/></X><Y1/></newConfigTree>", ConfigTree1.toString().trim());
	}
	
	public void test_rmvChildsByName() throws SAXException, IOException {
		ConfigTree ConfigTree1 = new ConfigTree("newConfigTree");
		ConfigTree ConfigTreeX = new ConfigTree("X",ConfigTree1);
		new ConfigTree("Y1",ConfigTree1);
		new ConfigTree("Y2",ConfigTreeX);
		
		assertEquals("<newConfigTree><X><Y2/></X><Y1/></newConfigTree>", ConfigTree1.toString().trim());
		ConfigTree1.removeChildrenByName("Y1");
		assertEquals("<newConfigTree><X><Y2/></X></newConfigTree>", ConfigTree1.toString().trim());
	}
	
	public void test_cloneObj() throws SAXException, IOException {
		ConfigTree ConfigTree1 = new ConfigTree("newConfigTree");
		new ConfigTree("X",ConfigTree1);
		new ConfigTree("Y",ConfigTree1);
		
		ConfigTree clone = ConfigTree1.cloneObj();
		
		assertTrue("Clone should produce a completely different object", (ConfigTree1 != clone));
		assertEquals("Clone should produce the same XML", ConfigTree1.toString(), clone.toString());
	}

    public void test_non_strings() throws SAXException {
        ConfigTree configTree = ConfigTree.fromXml("<config v1='11' v2='22.22' v3='true' v4='12x' />");

        assertEquals(11, configTree.getLongAttribute("v1", 123123));
        assertEquals(22.22f, configTree.getFloatAttribute("v2", 123123.123123f));
        assertEquals(true, configTree.getBooleanAttribute("v3", false));

        assertEquals(123123, configTree.getLongAttribute("v4", 123123));
        assertEquals(123123.123123f, configTree.getFloatAttribute("v4", 123123.123123f));
        assertEquals(false, configTree.getBooleanAttribute("v4", false));
    }
    
    public void test_getWholeText()
    {
        final ConfigTree configTree = new ConfigTree("test");
        new ConfigTree("child", configTree);
        try
        {
            configTree.getWholeText();
        }
        catch (final NullPointerException e)
        {
            fail("getWholeText should not throw NPE if child is empty: "  + e.getClass().getName());
        }
    }
}
