/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.routing.email;

import static org.junit.Assert.assertEquals;
import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.Email;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

/**
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class EmailerUnitTest
{
    @Test
    public void test_NotifyEmail() throws Exception 
    {
        ConfigTree config = new ConfigTree("email");
        
        config.setAttribute(Email.FROM, "a.b@c.com");
        config.setAttribute(Email.SENDTO, "d.e@f.com");
        config.setAttribute(Email.COPYTO, "g.h@i.com");
        config.setAttribute(Email.MESSAGE, "Hi there!!!");
        
        Emailer ne = new MockEmailer(config, "Hi there!!!\nHello");
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add("Hello".getBytes());
        ne.sendEmail(message);
    }

    @Test
    public void test_NotifyEmailNoMessage() throws Exception {
        ConfigTree config = new ConfigTree("email");
                
        config.setAttribute(Email.FROM, "a.b@c.com");
        config.setAttribute(Email.SENDTO, "d.e@f.com");
        config.setAttribute(Email.COPYTO, "g.h@i.com");
                
        Emailer ne = new MockEmailer(config, "Hello");
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add("Hello".getBytes());
        ne.sendEmail(message);
    }
    
    @Test
    public void overrideHost() throws Exception {
        assertOverride(Email.HOST, "b2.host2");
    }
    
    @Test
    public void overridePort() throws Exception {
        assertOverride(Email.PORT, "8021");
    }
    
    @Test
    public void overrideUsername() throws Exception {
        assertOverride(Email.USERNAME, "daniel");
    }
    
    @Test
    public void overridePassword() throws Exception {
        assertOverride(Email.PASSWORD, "letmein");
    }
    
    @Test
    public void overrideAuth() throws Exception {
        assertOverride(Email.AUTH, "true");
    }
    
    @Test
    public void overrideFrom() throws Exception {
        assertOverride(Email.FROM, "dbevenius@jboss.com");
    }
    
    @Test
    public void overrideSendTo() throws Exception {
        assertOverride(Email.SENDTO, "dbevenius@jboss.com");
    }
    
    @Test
    public void overrideCopyTo() throws Exception {
        assertOverride(Email.COPYTO, "dbevenius@jboss.com");
    }
    
    @Test
    public void overrideSubject() throws Exception {
        assertOverride(Email.SUBJECT, "Overridden Subject");
    }
    
    public void overrideAttachmentName() throws Exception {
        assertOverride("msgAttachmentName", "newFile.txt");
    }
    
    private void assertOverride(final String overrideName, final String overrideValue) throws Exception
    {
        ConfigTree emailMessageEl = new ConfigTree("email");
        emailMessageEl.setAttribute(Email.SENDTO, "d.e@f.com");
        emailMessageEl.setAttribute(Email.SUBJECT, "Original Subject");
        emailMessageEl.setAttribute(Email.MESSAGE, "Hi there!!!");
        
        MockEmailer ne = new MockEmailer(emailMessageEl, "Hi there!!!\nHello");
        Message message = MessageFactory.getInstance().getMessage();
        
        message.getProperties().setProperty(overrideName, overrideValue);
        message.getBody().add("Hello".getBytes());
        ne.sendEmail(message);
        
        assertEquals(overrideValue, ne.getConfigTree().getAttribute(overrideName));
        
    }
    
    private class MockEmailer extends Emailer 
    {
        private final String message ;
        
        private ConfigTree configTree;
        
        public MockEmailer(ConfigTree config, final String message) throws Exception 
        {
            super(config);
            this.message = message ;
        }
        
        @Override
        protected void sendEmail(ConfigTree messageParams, byte[] msgPayload)
        {
            configTree = messageParams;
            assertEquals(message, configTree.getAttribute(Email.MESSAGE));
        }

        public ConfigTree getConfigTree()
        {
            return configTree;
        }       
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(EmailerUnitTest.class);
    }
    
}

