/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.aggregation;

import junit.framework.TestCase;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.*;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.jboss.soa.esb.util.Util;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Tests for JBESB-1204 and JBESB-1331.
 * <p/>
 * Make sure the message aggregation info flows in the following scenario...
 * <pre>
 *
 *             |------------------ service1 ----------------|
 *             |                                            |
 * splitter1 --|                                            |-- aggregator2 --
 *             |               |--service2--|               |
 *             |-- splitter2 --|            |--aggregator1--|
 *                             |--service3--|
 *
 * </pre>
 *
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
public class Nested_Splits_UnitTest extends TestCase {

    private TestCourier splitter2Courier; // no need for a courier for splitter1
    private TestCourier service1Courier;
    private TestCourier service2Courier;
    private TestCourier service3Courier;
    private TestCourier aggregator1Courier;
    private TestCourier aggregator2Courier;

    private StaticRouter splitter1;
    private StaticRouter splitter2;
    private StaticRouter service1;
    private StaticRouter service2;
    private StaticRouter service3;
    private Aggregator aggregator1;
    private Aggregator aggregator2;
    private Aggregator aggregator3;

    protected void setUp() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        splitter2Courier = new TestCourier();
        service1Courier = new TestCourier();
        service2Courier = new TestCourier();
        service3Courier = new TestCourier();
        aggregator1Courier = new TestCourier();
        aggregator2Courier = new TestCourier();

        MockRegistry.register("test", "splitter2", splitter2Courier);
        MockRegistry.register("test", "service1", service1Courier);
        MockRegistry.register("test", "service2", service2Courier);
        MockRegistry.register("test", "service3", service3Courier);
        MockRegistry.register("test", "aggregator1", aggregator1Courier);
        MockRegistry.register("test", "aggregator2", aggregator2Courier);

        initaliseServices();
    }

    private void initaliseServices() throws ConfigurationException, RegistryException, ActionLifecycleException, IOException, SAXException {
        ESBConfigUtil esbConfig = new ESBConfigUtil(getClass().getResourceAsStream("action-configs-02.xml"));
        ConfigTree splitter1Config = esbConfig.getActionConfig("null-listener", "splitter1-action");
        ConfigTree splitter2Config = esbConfig.getActionConfig("null-listener", "splitter2-action");
        ConfigTree service1Config = esbConfig.getActionConfig("null-listener", "service1-config");
        ConfigTree service2Config = esbConfig.getActionConfig("null-listener", "service2-config");
        ConfigTree service3Config = esbConfig.getActionConfig("null-listener", "service3-config");
        ConfigTree aggregator1Config = esbConfig.getActionConfig("null-listener", "aggregator1-config");
        ConfigTree aggregator2Config = esbConfig.getActionConfig("null-listener", "aggregator2-config");
        ConfigTree aggregator3Config = esbConfig.getActionConfig("null-listener", "aggregator3-config");

        splitter1 = new StaticRouter(splitter1Config);
        splitter1.initialise();
        splitter2 = new StaticRouter(splitter2Config);
        splitter2.initialise();
        service1 = new StaticRouter(service1Config);
        service1.initialise();
        service2 = new StaticRouter(service2Config);
        service2.initialise();
        service3 = new StaticRouter(service3Config);
        service3.initialise();
        aggregator1 = new Aggregator(aggregator1Config);
        aggregator1.initialise();
        aggregator2 = new Aggregator(aggregator2Config);
        aggregator2.initialise();
        aggregator3 = new Aggregator(aggregator3Config);
        aggregator3.initialise();
    }

    protected void tearDown() throws Exception {
        splitter1.destroy();
        service1.destroy();
        service2.destroy();
        aggregator1.destroy();
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }

    /**
     *             |------------------ service1 ----------------|
     *             |                                            |
     * splitter1 --|                                            |-- aggregator2 --
     *             |               |--service2--|               |
     *             |-- splitter2 --|            |--aggregator1--|
     *                             |--service3--|
     */
    public void test_nested_split() throws RegistryException, ConfigurationException, ActionProcessingException, MessageDeliverException, IOException, SAXException, ParserConfigurationException {
        Message messageIn = MessageFactory.getInstance().getMessage();

        // Manually deliver the message to the splitter1 service...
        splitter1.process(messageIn);
        AggregationDetails service1Message = Aggregator.getAggregatorDetails(service1Courier.messages.get(0), 0, false);
        assertNotNull(service1Message);

        // Manually deliver the message in service1Courier into service1...
        service1.process(service1Courier.messages.get(0));

        // Manually deliver the message in splitter2Courier into splitter2...
        splitter2.process(splitter2Courier.messages.get(0));
        AggregationDetails service2Message = Aggregator.getAggregatorDetails(service2Courier.messages.get(0), 0, false);
        assertNotNull(service2Message);

        // Manually deliver the message in aggregator2Courier into aggregator2 (this is the message from service1)...
        aggregator2.process(aggregator2Courier.messages.get(0));

        // Manually deliver the message in service2Courier into service2...
        service2.process(service2Courier.messages.get(0));

        // Manually deliver the message in service3Courier into service3...
        service3.process(service3Courier.messages.get(0));

        // aggregator1Courier should have 2 messages in it... one from service2
        // and one from service3...
        assertEquals(2, aggregator1Courier.messages.size());

        // Should be 2 aggrTags before aggregation...
        List<String> aggrTags = Aggregator.getAggregatorTags(aggregator1Courier.messages.get(1), false);
        assertEquals(2, aggrTags.size());

        // Manually deliver the 2 messages to aggregator1...
        aggregator1.process(aggregator1Courier.messages.get(0));
        Message aggregator1Message = aggregator1.process(aggregator1Courier.messages.get(1));
        assertNoAggregationTags(aggregator1Courier.messages.get(0));
        assertNoAggregationTags(aggregator1Courier.messages.get(1));
        aggrTags = Aggregator.getAggregatorTags(aggregator1Message, false);
        assertEquals(1, aggrTags.size());
        AggregationDetails aggregator1MessageAggrDetails = Aggregator.getAggregatorDetails(aggregator1Message, 0, false);
        assertEquals(service1Message.getSeriesUuid(), aggregator1MessageAggrDetails.getSeriesUuid());

        // Manually deliver the aggregator1Message to aggregator2...
        Message aggregator2Message = aggregator2.process(aggregator1Message);
        assertNoAggregationTags(aggregator1Message);
        assertEquals(2, aggregator2Message.getAttachment().getUnnamedCount());
        Message message1 = Util.deserialize((Serializable) aggregator2Message.getAttachment().itemAt(0));
        assertNoAggregationTags(message1);
        Message message2 = Util.deserialize((Serializable) aggregator2Message.getAttachment().itemAt(1));
        assertNoAggregationTags(message2);
    }

    public void test_splitIds() throws ActionProcessingException {
        // Should be allowed to pump messages with diff splitIds into aggregator2
        // because there's no splitId set on it...
        test_splitIds(aggregator2, "splitId-1", "splitId-2");

        // Should get an error if I try the same with aggregator 3, coz it has
        // a "splitId" set on it...
        try {
            test_splitIds(aggregator3, "splitId-x", "splitId-y");
            fail("Expected ActionProcessingException");
        } catch (ActionProcessingException e) {
            assertEquals("Invalid aggregation config on aggregator 'aggregator3-config' .  This aggregator is configured to only aggregate message with an aggregation 'spliId' of 'splitId1'. The splitId on the received message is 'splitId-x'. A nested aggregation point may be missing, or may have been bypassed.", e.getMessage());
        }

        // The following should work because the splitIds match the split ID on
        // aggregator3's config (see action-configs-02.xml).
        test_splitIds(aggregator3, "splitId1", "splitId1");

    }

    private void test_splitIds(Aggregator aggregator, String splitId1, String splitId2) throws ActionProcessingException {
        Message messageIn = MessageFactory.getInstance().getMessage();
        AggregationDetails aggrDetails;
        List<String> aggrTags = new ArrayList<String>();

        aggrDetails = new AggregationDetails("xx", 1, 2, 123123);
        aggrDetails.setSplitId(splitId1);
        aggrTags.add(aggrDetails.toString());
        Aggregator.setAggregatorTags(messageIn, aggrTags, false);
        aggregator.process(messageIn);

        aggrDetails = new AggregationDetails("cc", 1, 2, 123123);
        aggrDetails.setSplitId(splitId1);
        aggrTags.add(aggrDetails.toString());
        Aggregator.setAggregatorTags(messageIn, aggrTags, false);
        aggregator.process(messageIn);

    }

    private void assertNoAggregationTags(Message aggregator1Message) {
        List<String> aggrTags;
        aggrTags = Aggregator.getAggregatorTags(aggregator1Message, false);
        assertNull(aggrTags);
    }
}