/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.aggregation;

import junit.framework.TestCase;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.actions.*;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.xml.sax.SAXException;

import java.util.List;
import java.io.IOException;

/**
 * Tests for JBESB-1204 and JBESB-1331 re aggrgation tag infos.
 * <p/>
 * Make sure the message aggregation info flows in the following scenario...
 * <pre>
 *
 *            |----- service1 -----|
 *            |                    |
 * splitter --|                    |-- aggregator
 *            |                    |
 *            |----- service2 -----|
 *
 * </pre>
 *
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
public class JBESB_1204_1331_UnitTest extends TestCase {

    private TestCourier service1Courier;
    private TestCourier service2Courier;
    private TestCourier aggregatorCourier;

    private StaticRouter splitter;
    private StaticRouter service1;
    private StaticRouter service2;
    private Aggregator aggregator;

    protected void setUp() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        service1Courier = new TestCourier();
        service2Courier = new TestCourier();
        aggregatorCourier = new TestCourier();

        MockRegistry.register("test", "service1", service1Courier);
        MockRegistry.register("test", "service2", service2Courier);
        MockRegistry.register("test", "aggregator", aggregatorCourier);

        initaliseServices();
    }

    private void initaliseServices() throws ConfigurationException, RegistryException, ActionLifecycleException, IOException, SAXException {
        ESBConfigUtil esbConfig = new ESBConfigUtil(getClass().getResourceAsStream("action-configs-01.xml"));
        ConfigTree splitterConfig = esbConfig.getActionConfig("null-listener", "splitter1-action");
        ConfigTree service1Config = esbConfig.getActionConfig("null-listener", "service1-config");
        ConfigTree service2Config = esbConfig.getActionConfig("null-listener", "service2-config");
        ConfigTree aggregatorConfig = esbConfig.getActionConfig("null-listener", "aggregator-config");

        splitter = new StaticRouter(splitterConfig);
        splitter.initialise();
        service1 = new StaticRouter(service1Config);
        service1.initialise();
        service2 = new StaticRouter(service2Config);
        service2.initialise();
        aggregator = new Aggregator(aggregatorConfig);
        aggregator.initialise();
    }

    protected void tearDown() throws Exception {
        splitter.destroy();
        service1.destroy();
        service2.destroy();
        aggregator.destroy();
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }

    public void test() throws RegistryException, ConfigurationException, ActionProcessingException, MessageDeliverException {
        Message messageIn = MessageFactory.getInstance().getMessage();
        List<String> aggrTags;
        AggregationDetails aggrDetailsS1;
        AggregationDetails aggrDetailsS2;
        AggregationDetails aggrDetailsAggrM1;
        AggregationDetails aggrDetailsAggrM2;

        // Manually deliver the message to the splitter service...
        splitter.process(messageIn);
        
        // There should be 2 msssages delivered.. one to service1 and one service2...
        assertEquals(1, service1Courier.messages.size());
        assertEquals(1, service2Courier.messages.size());

        // The 2 messages should each have 1 aggregation info string (no more)...
        aggrTags = Aggregator.getAggregatorTags(service1Courier.messages.get(0), false);
        assertEquals(1, aggrTags.size());
        aggrDetailsS1 = new AggregationDetails(aggrTags.get(0));
        aggrTags = Aggregator.getAggregatorTags(service2Courier.messages.get(0), false);
        assertEquals(1, aggrTags.size());
        aggrDetailsS2 = new AggregationDetails(aggrTags.get(0));        

        // Manually deliver the message in service1Courier into service1...
        service1.process(service1Courier.messages.get(0));

        // Manually deliver the message in service2Courier into service2...
        service2.process(service2Courier.messages.get(0));

        // 2 messages should arrive at the aggregatorCourier...
        assertEquals(2, aggregatorCourier.messages.size());

        // Extract aggr details from first message...
        aggrTags = Aggregator.getAggregatorTags(aggregatorCourier.messages.get(0), false);
        assertEquals(1, aggrTags.size());
        aggrDetailsAggrM1 = new AggregationDetails(aggrTags.get(0));

        // Extract aggr details from second message...
        aggrTags = Aggregator.getAggregatorTags(aggregatorCourier.messages.get(1), false);
        assertEquals(1, aggrTags.size());
        aggrDetailsAggrM2 = new AggregationDetails(aggrTags.get(0));

        // make sure all the UUIDs match...
        assertEquals(aggrDetailsS1.getSeriesUuid(), aggrDetailsS2.getSeriesUuid()); 
        assertEquals(aggrDetailsS2.getSeriesUuid(), aggrDetailsAggrM1.getSeriesUuid());
        assertEquals(aggrDetailsAggrM1.getSeriesUuid(), aggrDetailsAggrM2.getSeriesUuid()); 

        // make sure all the timestamps match...
        assertEquals(aggrDetailsS1.getSeriesTimestamp(), aggrDetailsS2.getSeriesTimestamp());
        assertEquals(aggrDetailsS2.getSeriesTimestamp(), aggrDetailsAggrM1.getSeriesTimestamp());
        assertEquals(aggrDetailsAggrM1.getSeriesTimestamp(), aggrDetailsAggrM2.getSeriesTimestamp()); 

        // make sure the series size matches...
        assertEquals(aggrDetailsS1.getSeriesSize(), aggrDetailsS2.getSeriesSize());
        assertEquals(aggrDetailsS2.getSeriesSize(), aggrDetailsAggrM1.getSeriesSize());
        assertEquals(aggrDetailsAggrM1.getSeriesSize(), aggrDetailsAggrM2.getSeriesSize());

        // make sure the message num's don't match, and that they're 1 or 2...
        assertNotSame(aggrDetailsAggrM1.getMessageNumber(), aggrDetailsAggrM2.getMessageNumber());
        assertTrue(aggrDetailsAggrM1.getMessageNumber() == 1 || aggrDetailsAggrM1.getMessageNumber() == 2);
        assertTrue(aggrDetailsAggrM2.getMessageNumber() == 1 || aggrDetailsAggrM2.getMessageNumber() == 2);

        // make sure the splitId matches...
        assertEquals("splitter1-action", aggrDetailsS1.getSplitId());
        assertEquals(aggrDetailsS1.getSplitId(), aggrDetailsS2.getSplitId());
        assertEquals(aggrDetailsS2.getSplitId(), aggrDetailsAggrM1.getSplitId());
        assertEquals(aggrDetailsAggrM1.getSplitId(), aggrDetailsAggrM2.getSplitId());
    }
}
