/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions;

import junit.framework.TestCase;
import org.jboss.soa.esb.testutils.AbstractTestRunner;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.internal.soa.esb.rosetta.pooling.MockTransactionStrategy;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SyncServiceInvokerUnitTest extends TestCase {

    protected void setUp() throws Exception {
        ResponseAction.responseMessage = null;
        ResponseAction.exception = null;
    }

    public void test_OK() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("Services", "ServiceA");
                Message request = MessageFactory.getInstance().getMessage();

                request.getBody().add("Hello");

                ResponseAction.responseMessage = MessageFactory.getInstance().getMessage();
                ResponseAction.responseMessage.getBody().add("Goodbye");

                Message response = invoker.deliverSync(request, 10000);
                assertEquals("Goodbye", response.getBody().get());
            }
        }.setServiceConfig("sync-invoker-config-01.xml");

        testRunner.run();
    }

    public void test_exception_fail() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("Services", "ServiceA");
                Message request = MessageFactory.getInstance().getMessage();

                request.getBody().add("Hello");

                ResponseAction.exception = new ActionProcessingException("Exception!!");

                try {
                    invoker.deliverSync(request, 10000);
                    fail("Expected FaultMessageException");
                } catch(FaultMessageException e) {
                    assertEquals("Error delivering message to service 'Services:ServiceB'", e.getCause().getMessage());
                }
            }
        }.setServiceConfig("sync-invoker-config-01.xml");

        testRunner.run();
    }

    public void test_exception_OK() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("Services", "ServiceA");
                Message request = MessageFactory.getInstance().getMessage();

                request.getBody().add("Hello");

                ResponseAction.exception = new ActionProcessingException("Exception!!");

                invoker.deliverSync(request, 10000);
            }
        }.setServiceConfig("sync-invoker-config-02.xml");

        testRunner.run();
    }

    public void x_test_transaction_not_suspend() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                SuspendedTransactionStrategy suspendedTxStrategy = new SuspendedTransactionStrategy();

                TransactionStrategy.setTransactionStrategy(suspendedTxStrategy);
                try {
                    ServiceInvoker invoker = new ServiceInvoker("Services", "ServiceA");
                    Message request = MessageFactory.getInstance().getMessage();
                    ResponseAction.responseMessage = MessageFactory.getInstance().getMessage();

                    request.getBody().add("Hello");

                    // Need to deliver async here because we're in a transaction (mock transaction)...
                    invoker.deliverAsync(request);
                    ResponseAction.waitForMessage(10000);
                } finally {
                    TransactionStrategy.setTransactionStrategy(new TransactionStrategy.NullTransactionStrategy());
                }

                assertFalse(suspendedTxStrategy.suspended);
                assertFalse(suspendedTxStrategy.resumed);
            }
        }.setServiceConfig("sync-invoker-config-01.xml");

        testRunner.run();
    }

    public void x_test_transaction_suspend() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                SuspendedTransactionStrategy suspendedTxStrategy = new SuspendedTransactionStrategy();

                TransactionStrategy.setTransactionStrategy(suspendedTxStrategy);
                try {
                    ServiceInvoker invoker = new ServiceInvoker("Services", "ServiceA");
                    Message request = MessageFactory.getInstance().getMessage();
                    ResponseAction.responseMessage = MessageFactory.getInstance().getMessage();

                    request.getBody().add("Hello");

                    // Need to deliver async here because we're in a transaction (mock transaction)...
                    invoker.deliverAsync(request);
                    ResponseAction.waitForMessage(10000);
                } finally {
                    TransactionStrategy.setTransactionStrategy(new TransactionStrategy.NullTransactionStrategy());
                }

                assertTrue(suspendedTxStrategy.suspended);
                assertTrue(suspendedTxStrategy.resumed);
            }
        }.setServiceConfig("sync-invoker-config-03.xml");

        testRunner.run();
    }

    private class SuspendedTransactionStrategy extends MockTransactionStrategy {
        private boolean suspended = false;
        private boolean resumed = false;

        private SuspendedTransactionStrategy() {
            reset();
        }

        public Object suspend() throws TransactionStrategyException {
            suspended = true;
            return super.suspend();
        }

        public void resume(Object tx) throws TransactionStrategyException {
            resumed = true;
            super.resume(tx);
        }
    }
}
