/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.webservice;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.listeners.config.Generator;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryFactory;
import org.jboss.soa.esb.util.ClassUtil;

public class JAXWSProviderClassGeneratorUnitTest extends TestCase
{
    private TestRegistry testRegistry ;
    
    @Override
    protected void setUp()
        throws Exception
    {
        MockCourierFactory.install() ;
        testRegistry = new TestRegistry() ;
        RegistryFactory.setRegistry(testRegistry) ;
    }
    
    @Override
    protected void tearDown()
        throws Exception
    {
        RegistryFactory.setRegistry(null) ;
        MockCourierFactory.uninstall() ;
    }
    
    public void testRegistryLookupCount()
        throws Exception
    {
        final InputStream is = ClassUtil.getResourceAsStream("jbossesb_config_08.xml", getClass()) ;
        final byte[] configBytes = StreamUtils.readStream(is) ;
        ByteArrayOutputStream listenerXml = new ByteArrayOutputStream();
        ByteArrayOutputStream gatewayXml = new ByteArrayOutputStream();
        Generator generator = new Generator(new ByteArrayInputStream(configBytes), listenerXml, gatewayXml);
        generator.generate();
        List<WebserviceInfo> webservices = generator.getModel().getWebserviceServices();
        final WebserviceInfo webservice = webservices.get(0) ;
        final ESBServiceEndpointInfo endpointInfo = new ESBServiceEndpointInfo(webservice) ;
        
        final EPR deliverEPR = new EPR(new URI("test:deliver")) ;
        final MockCourier deliverCourier = new MockCourier(true);
        final Service service = webservice.getService() ;
        final String category = service.getCategory() ;
        final String name = service.getName() ;
        TestRegistry.register(category, name, deliverEPR, deliverCourier);

        final JAXWSProviderClassGenerator classGenerator = new JAXWSProviderClassGenerator() ;
        
        final byte[] bytes = classGenerator.generate("deploymentName", category, name, endpointInfo, false) ;
        final ClassLoader classLoader = new ClassLoader() {
            public java.lang.Class<?> loadClass(final String name) throws ClassNotFoundException {
                if (endpointInfo.getClassName().equals(name)) {
                    return defineClass(name, bytes, 0, bytes.length) ;
                } else {
                    return super.loadClass(name) ;
                }
            };
        } ;
        final Class<?> baseWebServiceClass = classLoader.loadClass(endpointInfo.getClassName()) ;
        
        final BaseWebService baseWebService = (BaseWebService)baseWebServiceClass.newInstance() ;
        assertEquals("Initial instance count", 1, testRegistry.getCount(category, name)) ;
        
        final Message message = MessageFactory.getInstance().getMessage() ;
        baseWebService.deliverMessage(message) ;
        assertEquals("First invocation", 1, testRegistry.getCount(category, name)) ;
        
        final BaseWebService baseWebService2 = (BaseWebService)baseWebServiceClass.newInstance() ;
        assertEquals("Second instance count", 1, testRegistry.getCount(category, name)) ;
        
        baseWebService2.deliverMessage(message) ;
        assertEquals("Second invocation", 1, testRegistry.getCount(category, name)) ;
    }
    
    final class TestRegistry extends MockRegistry
    {
        final Map<Service, Integer> counts = new HashMap<Service, Integer>() ;
        
        @Override
        public List<EPR> findEPRs(final String category, final String serviceName)
            throws RegistryException
        {
            final Service service = new Service(category, serviceName) ;
            final Integer count = counts.get(service) ;
            if (count != null)
            {
                counts.put(service, Integer.valueOf(count.intValue() + 1)) ;
            }
            else
            {
                counts.put(service, Integer.valueOf(1)) ;
            }
            
            return super.findEPRs(category, serviceName) ;
        }
        
        public int getCount(final String category, final String serviceName)
        {
            final Service service = new Service(category, serviceName) ;
            final Integer count = counts.get(service) ;
            if (count != null)
            {
                return count.intValue() ;
            }
            return 0 ;
        }
    }
}
