/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. 
 * See the copyright.txt in the distribution for a full listing 
 * of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License, v. 2.0.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License,
 * v. 2.0 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.internal.soa.esb.util;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertNotNull;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.Properties;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.stream.StreamResult;
import javax.xml.validation.Schema;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Tests associated with the XML helper class.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */
public class XMLHelperUnitTest
{
    @Test
    public void testReplaceSystemProperties()
        throws Exception
    {
        final Properties props = System.getProperties() ;
        props.setProperty("test.property.attributeValue1", " testAttributeValue1 ") ;
        props.setProperty("test.property.attributeValue2", "testAttributeValue2") ;
        props.setProperty("test.property.attributeValue3", "${test.property.attributeValue1}") ;
        props.setProperty("test.property.attributeValue4", "<testAttr/>") ;
        props.setProperty("test.property.textValue1", "testTextValue1") ;
        props.setProperty("test.property.textValue2", "${test.property.textValue2}") ;
        props.setProperty("test.property.textValue3", "<testText/>") ;
        
        final InputStream is = getClass().getResourceAsStream("replaceSystemProperties_source.xml") ;
        final XMLStreamReader streamReader = XMLHelper.getXMLStreamReader(is) ;
        final String encoding = streamReader.getEncoding() ;
        
        final StringWriter sw = new StringWriter() ;
        
        XMLHelper.replaceSystemProperties(streamReader,
            XMLHelper.getXMLStreamWriter(sw)) ;
        final String contents = sw.toString() ;
        
        final String expectedContents = StreamUtils.getResourceAsString("replaceSystemProperties_expected.xml", encoding) ;

        final boolean match = XMLHelper.compareXMLContent(expectedContents, contents) ;
        assertTrue("System property replacement", match) ;
    }
    
    @Test
    public void testValidateSchemaOnCPWithImportsFromSubdir() throws ConfigurationException, UnsupportedEncodingException, SAXException
    {
        final Schema schema = XMLHelper.getSchema("/org/jboss/internal/soa/esb/util/schemas/request.xsd");
        
        final String xml = StreamUtils.readStreamString(getClass().getResourceAsStream("sample-request.xml"), "UTF-8");
        assertTrue(XMLHelper.validate(schema, xml));
    }
    
    @Test
    public void testValidateSchemaOnCPWithImports() throws ConfigurationException, UnsupportedEncodingException, SAXException
    {
        final Schema schema = XMLHelper.getSchema("/org/jboss/internal/soa/esb/util/request2.xsd");
        
        final String xml = StreamUtils.readStreamString(getClass().getResourceAsStream("sample-request.xml"), "UTF-8");
        assertTrue(XMLHelper.validate(schema, xml));
    }
    
    @Test
    public void testValidateSchemaOnFSWithImports() throws ConfigurationException, UnsupportedEncodingException, SAXException
    {
        URL resource = getClass().getResource("/org/jboss/internal/soa/esb/util/request2.xsd");
        File file = new File(resource.getFile());
        final Schema schema = XMLHelper.getSchema(file.getAbsolutePath());
        
        final String xml = StreamUtils.readStreamString(getClass().getResourceAsStream("sample-request.xml"), "UTF-8");
        assertTrue(XMLHelper.validate(schema, xml));
    }
    
    @Test
    public void testCopyWithDeclaration()
        throws Exception
    {
        final InputStream is = ClassUtil.getResourceAsStream("copy_source.xml", getClass()) ;
        final XMLEventReader reader = XMLHelper.getXMLEventReader(is) ;
        final StringWriter sw = new StringWriter() ;
        final XMLEventWriter writer = XMLHelper.getXMLEventWriter(sw) ;
        XMLHelper.copyXMLEventStream(reader, writer) ;
        final String copy = sw.toString() ;
        assertTrue("Copy does not contain declaration", copy.contains("<?xml")) ;
    }
    
    @Test
    public void testCopyWithoutDeclaration()
        throws Exception
    {
        final InputStream is = ClassUtil.getResourceAsStream("copy_source.xml", getClass()) ;
        final XMLEventReader reader = XMLHelper.getXMLEventReader(is) ;
        final StringWriter sw = new StringWriter() ;
        final XMLEventWriter writer = XMLHelper.getXMLEventWriter(sw) ;
        XMLHelper.copyXMLEventStream(reader, writer, true) ;
        final String copy = sw.toString() ;
        assertFalse("Copy contains declaration", copy.contains("<?xml")) ;
    }

    @Test
    public void testSubDocumentParsing()
        throws Exception
    {
        final InputStream is = ClassUtil.getResourceAsStream("copy_source.xml", getClass()) ;
        final Document doc = YADOMUtil.parseStream(is, false, false, true) ;
        final org.w3c.dom.Element root = doc.getDocumentElement() ;
        
        org.w3c.dom.Element body = null ;

        final NodeList list = root.getChildNodes() ;
        final int numChildren = list.getLength() ;
        for(int count = 0 ; count < numChildren ; count++)
        {
            final org.w3c.dom.Node node = list.item(count) ;
            if (node instanceof Element)
            {
                final org.w3c.dom.Element child = (org.w3c.dom.Element)node ;
                if ("Body".equals(child.getLocalName()))
                {
                    body = child ;
                }
            }
        }
        
        assertNotNull("body element is null", body) ;
        
        final StringWriter sw = new StringWriter() ;
        final XMLEventWriter writer = XMLHelper.getXMLEventWriter(new StreamResult(sw)) ;
        XMLHelper.readDomNode(body, writer, true) ;
        
        final String bodyVal = sw.toString() ;
        
        final XMLEventReader reader = XMLHelper.getXMLEventReader(new ByteArrayInputStream(bodyVal.getBytes())) ;
        final Document document = XMLHelper.createDocument(reader) ;
        assertNotNull("document is null", document) ;
    }


    @Test
    public void testWhitespaceParsing()
        throws Exception
    {
        final InputStream is = ClassUtil.getResourceAsStream("whitespace_source.xml", getClass()) ;
        final XMLEventReader reader = XMLHelper.getXMLEventReader(is) ;
        final Document document = XMLHelper.createDocument(reader) ;
        
        assertNotNull("document is null", document) ;
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(XMLHelperUnitTest.class);
    }
    
}
