/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.security;

import static org.junit.Assert.*;

import javax.security.auth.Subject;

import org.jboss.internal.soa.esb.services.security.JBossASContextPropagator.AS4SecurityContextOperations;
import org.jboss.internal.soa.esb.services.security.JBossASContextPropagator.AS5SecurityContextOperations;
import org.jboss.internal.soa.esb.services.security.JBossASContextPropagator.JBossASSecurityContextOperations;
import org.jboss.security.SecurityAssociation;
import org.jboss.security.SecurityContextAssociation;
import org.jboss.security.SubjectInfo;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.services.security.SecurityConfig;
import org.jboss.soa.esb.services.security.SecurityConfigTestUtil;
import org.jboss.soa.esb.services.security.SecurityConfigUtil;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.principals.User;
import org.junit.Test;

import junit.framework.JUnit4TestAdapter;

/**
 * Unit test for {@link JBossASContextPropagator}.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class JBossASContextPropagatorUnitTest
{
    private JBossASContextPropagator propagator = new JBossASContextPropagator();
    private JBossASSecurityContextOperations as4Operations = new AS4SecurityContextOperations() ;
    private JBossASSecurityContextOperations as5Operations = new AS5SecurityContextOperations() ;

    @Test
    public void pushAndPop()
    {
        final SecurityConfig securityConfig = getSecurityConfig();
        final SecurityContext context = getSecurityContext();

        propagator.pushSecurityContext(context, null, securityConfig, as4Operations);

        Subject pushedSubject = SecurityAssociation.getSubject();
        assertEquals(context.getSubject(), pushedSubject);

        propagator.popSecurityContext(context, securityConfig, as4Operations);
        pushedSubject = SecurityAssociation.getSubject();
        assertNull(pushedSubject);
    }

    @Test
    public void pushAndPopWithNullSecurityConfig()
    {
        final SecurityConfig securityConfig = getSecurityConfig();
        final SecurityContext context = getSecurityContext();

        propagator.pushSecurityContext(context, null, null, as4Operations);

        Subject pushedSubject = SecurityAssociation.getSubject();
        assertEquals(context.getSubject(), pushedSubject);

        propagator.popSecurityContext(context, securityConfig, as4Operations);
        pushedSubject = SecurityAssociation.getSubject();
        assertNull(pushedSubject);
    }

    @Test
    public void popWithNullSecurityConfig()
    {
        propagator.popSecurityContext(null, null, as4Operations);
    }

    @Test
    public void pushAndPopAS5()
    {
        final SecurityConfig securityConfig = getSecurityConfig();
        final SecurityContext context = getSecurityContext();

        propagator.pushSecurityContext(context, null, securityConfig, as5Operations);

        final org.jboss.security.SecurityContext securityContext = SecurityContextAssociation.getSecurityContext() ;
        assertNotNull("securityContext", securityContext) ;
        
        final SubjectInfo subjectInfo = securityContext.getSubjectInfo() ;
        assertNotNull("subjectInfo", subjectInfo) ;
        
        final Subject pushedSubject = subjectInfo.getAuthenticatedSubject() ;
        assertEquals("Subject comparison", context.getSubject(), pushedSubject);

        propagator.popSecurityContext(context, securityConfig, as5Operations);
        assertNull("securityContext", SecurityContextAssociation.getSecurityContext());
    }

    @Test
    public void pushAndPopWithNullSecurityConfigAS5()
    {
        final SecurityConfig securityConfig = getSecurityConfig();
        final SecurityContext context = getSecurityContext();

        propagator.pushSecurityContext(context, null, null, as5Operations);

        final org.jboss.security.SecurityContext securityContext = SecurityContextAssociation.getSecurityContext() ;
        assertNotNull("securityContext", securityContext) ;
        
        final SubjectInfo subjectInfo = securityContext.getSubjectInfo() ;
        assertNotNull("subjectInfo", subjectInfo) ;
        
        final Subject pushedSubject = subjectInfo.getAuthenticatedSubject() ;
        assertEquals("Subject comparison", context.getSubject(), pushedSubject);

        propagator.popSecurityContext(context, securityConfig, as5Operations);
        assertNull("securityContext", SecurityContextAssociation.getSecurityContext());
    }

    @Test
    public void popWithNullSecurityConfigAS5()
    {
        propagator.popSecurityContext(null, null, as5Operations);
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(JBossASContextPropagatorUnitTest.class);
    }

    private SecurityContext getSecurityContext()
    {
        Subject subject = new Subject();
        subject.getPrincipals().add(new User("Mr.Poon"));
        return new SecurityContext(subject, 5000l);
    }

    private SecurityConfig getSecurityConfig()
    {
        ConfigTree securityFragment = SecurityConfigTestUtil.createSecurityFragment("adminRole", null, "SuccessfulLogin");
        return SecurityConfigUtil.createSecurityConfig(securityFragment);
    }
}
