/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.services.registry;

import java.util.List;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryInterceptor;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;


/**
 * Concurrency test interceptor.
 * 
 * @author <a href="mailto:Kevin.Conner@jboss.com">Kevin Conner</a>
 */
class RegistryConcurrencyInterceptor implements RegistryInterceptor
{
    private final CyclicBarrier barrier ;

    private volatile boolean timeout ;
    private Registry registry ;
    private final AtomicInteger findEPRsCount = new AtomicInteger() ;

    public RegistryConcurrencyInterceptor(final CyclicBarrier barrier)
    {
        this.barrier = barrier ;
    }

    public List<String> findAllServices()
        throws RegistryException
    {
        return registry.findAllServices() ;
    }

    public EPR findEPR(final String serviceCategoryName, final String serviceName)
        throws RegistryException, ServiceNotFoundException
    {
        return registry.findEPR(serviceCategoryName, serviceName) ;
    }

    public List<EPR> findEPRs(final String serviceCategoryName, final String serviceName)
        throws RegistryException, ServiceNotFoundException
    {
        findEPRsCount.incrementAndGet() ;
        if (!timeout)
        {
            try
            {
                barrier.await(10, TimeUnit.SECONDS) ;
            }
            catch (final TimeoutException te)
            {
                timeout = true ;
            }
            catch (final InterruptedException ie)
            {
                throw new RegistryException("Interrupted", ie) ;
            }
            catch (final BrokenBarrierException bbe)
            {
                throw new RegistryException("Broken barrier", bbe) ;
            }
        }
        return registry.findEPRs(serviceCategoryName, serviceName) ;
    }

    public int getFindEPRsCount()
    {
        return findEPRsCount.get() ;
    }

    public boolean isTimeout()
    {
        return timeout ;
    }

    public List<String> findServices(final String serviceCategoryName)
        throws RegistryException
    {
        return registry.findServices(serviceCategoryName) ;
    }

    public void registerEPR(final String serviceCategoryName, final String serviceName,
        final String serviceDescription, final EPR epr, final String eprDescription)
        throws RegistryException
    {
        registry.registerEPR(serviceCategoryName, serviceName, serviceDescription, epr, eprDescription) ;
    }

    public void unRegisterEPR(final String serviceCategoryName,
        final String serviceName, final EPR epr)
        throws RegistryException, ServiceNotFoundException
    {
        registry.unRegisterEPR(serviceCategoryName, serviceName, epr) ;
    }

    public void unRegisterService(final String category, final String serviceName)
        throws RegistryException, ServiceNotFoundException
    {
        registry.unRegisterService(category, serviceName) ;
    }

    public void setRegistry(final Registry registry)
    {
        this.registry = registry ;
    }
}
