/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.services.registry;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;

import junit.framework.TestCase;	

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.Connection;
import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.JAXRResponse;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.Service;
import javax.xml.registry.infomodel.ServiceBinding;
import javax.xml.registry.infomodel.SpecificationLink;
import javax.xml.registry.infomodel.User;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.ws.scout.registry.infomodel.SpecificationLinkImpl;
import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.MarshalException;

import org.junit.AfterClass;
import org.junit.BeforeClass;

/**
 * Testing the ability of the JAXRRegistryImpl to coalesce duplicate
 * services into an aggregate service.     EPRs from the duplicate
 * service entries are copied to the aggregate service, and the duplicate entries
 * are deleted.
 * 
 * @author <a href="tcunning@redhat.com">Tom Cunningham</a>
 */
public class JAXRCoalesceUnitTest extends TestCase	
{
	private static Logger logger = Logger.getLogger(JAXRCoalesceUnitTest.class);
	
	private ClassificationScheme jbossESBScheme;

	private static final String TEST_SERVICE_CATEGORY = "ESB_CATEGORY";
	private static final String TEST_SERVICE_NAME = "SERVICE_NAME";
	private static final String TEST_SERVICE_DESC = "SERVICE_DESCRIPTION";
	
	private static boolean startflag = false;
	
	/**
	 * Setup the database.
	 * @throws Exception
	 */
	public void setUp() throws Exception {
		if (!startflag) {
			TestEnvironmentUtil.setESBPropertiesFileToUse();
			TestEnvironmentUtil.startJAXRDatabase();
			startflag = true;
		}
	}
	
	@AfterClass
	public void afterClass() throws Exception {
		TestEnvironmentUtil.stopJAXRDatabase();
		startflag = false;
	}

	public static void scenarioOne() throws Exception {
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		Service s1 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME, TEST_SERVICE_DESC);
		Service s2 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME, TEST_SERVICE_DESC);	
		
		// Attach EPRs
		InVMEpr epr1 = new InVMEpr(new URI("invm://service1epr1"));
		InVMEpr epr2 = new InVMEpr(new URI("invm://service1epr2"));
		registerEPR(registry, s1, epr1, "epr1");
		registerEPR(registry, s1, epr2, "epr2");

		InVMEpr epr3 = new InVMEpr(new URI("invm://service2epr1"));
		InVMEpr epr4 = new InVMEpr(new URI("invm://service2epr2"));
		registerEPR(registry, s2, epr3, "epr3");
		registerEPR(registry, s2, epr4, "epr4");		
	}

	public static void scenarioTwo() throws Exception {
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		Service s1 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME+"2", TEST_SERVICE_DESC);
		Service s2 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME+"2", TEST_SERVICE_DESC);	
		
		// Attach EPRs
		InVMEpr epr1 = new InVMEpr(new URI("invm://service1epr1"));
		InVMEpr epr2 = new InVMEpr(new URI("invm://service1epr2"));
		registerEPR(registry, s1, epr1, "epr1");
		registerEPR(registry, s1, epr2, "epr2");

		InVMEpr epr3 = new InVMEpr(new URI("invm://service1epr1"));
		InVMEpr epr4 = new InVMEpr(new URI("invm://service1epr2"));
		registerEPR(registry, s2, epr3, "epr1");
		registerEPR(registry, s2, epr4, "epr2");		
	}

	public static void scenarioThree() throws ConfigurationException, URISyntaxException, JAXRException, RegistryException {
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		Service s1 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME+"3", TEST_SERVICE_DESC);
		Service s2 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME+"3", TEST_SERVICE_DESC);	
		
		InVMEpr epr3 = new InVMEpr(new URI("invm://service1epr1"));
		InVMEpr epr4 = new InVMEpr(new URI("invm://service1epr2"));
		registerEPR(registry, s2, epr3, "epr1");
		registerEPR(registry, s2, epr4, "epr2");		
	}

	public static void scenarioFour() throws ConfigurationException, URISyntaxException, JAXRException, RegistryException {
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		Service s1 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME+"4", TEST_SERVICE_DESC);
		Service s2 = registry.registerService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME+"4second", TEST_SERVICE_DESC);
		InVMEpr epr3 = new InVMEpr(new URI("invm://service1epr1"));
		InVMEpr epr4 = new InVMEpr(new URI("invm://service1epr2"));
		registerEPR(registry, s1, epr3, "epr1");
		registerEPR(registry, s1, epr4, "epr2");		
	}
	
	/** 
	 * Publish an EPR to the Registry
	 */
	public static void registerEPR(JAXRRegistryImpl registry, Service service, EPR epr, String eprDescription) 
		throws RegistryException, ConfigurationException
	{
		if (epr == null)
		{
			throw new RegistryException("Null EPR argument specified") ;
		}
		JAXRConnectionFactory jaxrConnectionFactory = new JAXRConnectionFactory();
		Connection connection = jaxrConnectionFactory.getConnection();
        try {
			final Concept jbossTModel = registry.getJBossESBTModel(connection);

			RegistryService rs = connection.getRegistryService();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            BusinessQueryManager bqm = rs.getBusinessQueryManager();
            ServiceBinding serviceBinding = blm.createServiceBinding();
            String xml = EPRHelper.toXMLString(epr);
            serviceBinding.setAccessURI(xml);
            
            ArrayList<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
            SpecificationLink sl = new SpecificationLinkImpl(blm);
            sl.setSpecificationObject(jbossTModel);
            serviceBinding.getSpecificationLinks().add(sl);
            if ("3.0".equals(Configuration.getRegistryUDDIVersion())) {
               if ((eprDescription == null) || ("".equals(eprDescription))) {
                   serviceBinding.setDescription(blm.createInternationalString(JAXRRegistryImpl.REGISTERED_BY_JBOSSESB));
               }
            } else {
               serviceBinding.setDescription(blm.createInternationalString(eprDescription));
            }
            serviceBindings.add(serviceBinding);
            
            service.addServiceBindings(serviceBindings);
            ClassificationScheme cScheme = registry.getClassificationScheme(bqm, blm);
            Classification classification = blm.createClassification(cScheme, "category", TEST_SERVICE_CATEGORY);
            service.addClassification(classification);
            
            registry.saveRegistryObject(serviceBinding, jaxrConnectionFactory);
        } catch (JAXRException je) {
            throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (MarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
        } finally {
            jaxrConnectionFactory.closeConnection(connection);
        }
	}
	
	public Collection findMultipleServices(String category, String name) throws ConfigurationException, JAXRException, RegistryException {
		JAXRConnectionFactory jaxrConnectionFactory = new JAXRConnectionFactory();
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			JAXRRegistryImpl registry = new JAXRRegistryImpl();
			
			final Concept jbossTModel = registry.getJBossESBTModel(connection);
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            final ClassificationScheme cScheme = registry.getClassificationScheme(bqm, blm) ;
            Collection<Classification> classifications = new ArrayList<Classification>();
            Classification classification =
                blm.createClassification(
                  cScheme,
                  "category", category);
            classifications.add(classification);
			Collection<String> namePatterns = new ArrayList<String>();
			namePatterns.add(name);

            Collection<SpecificationLink> specifications = new ArrayList<SpecificationLink>();
            
	        //Find based upon qualifier type and values
			Collection<String> findQualifiers = new ArrayList<String>();
			findQualifiers.add(FindQualifier.AND_ALL_KEYS);
			findQualifiers.add(FindQualifier.EXACT_NAME_MATCH);
			findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
			logger.log(Level.DEBUG, "Going to query the registry for name pattern " + namePatterns);
			BulkResponse response = bqm.findServices(null, findQualifiers,
					namePatterns, classifications, specifications);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				return response.getCollection();
			} else {
				throw new RegistryException();
			}
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}
	
	/*
	 * testScenarioOne tests a scenario where two duplicate services each have
	 * two unique EPRs.    The result is that the EPRs from the second service
	 * are merged over to the first.
	 */
	public void testScenarioOne() throws ConfigurationException, JAXRException, RegistryException, URISyntaxException, Exception
	{
		scenarioOne();
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		registry.findService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME);
		Collection services = findMultipleServices(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME);
		assertEquals(1, services.size());

		for (Iterator iter = services.iterator(); iter.hasNext();) {
			Service s = (Service) iter.next();
			assertEquals(s.getName().getValue(), TEST_SERVICE_NAME);
			assertEquals(s.getDescription().getValue(), TEST_SERVICE_DESC);
			assertEquals(4, s.getServiceBindings().size());
		}

	}

	/*
	 * testScenarioTwo tests a scenario where two duplicate services each
	 * have the same two EPRs.    The result should be that the second service
	 * and the duplicate EPRs are removed and you have the original service
	 * with the original EPRs.
	 */
	public void testScenarioTwo() throws ConfigurationException, JAXRException, RegistryException, URISyntaxException, Exception
	{
		scenarioTwo();
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		registry.findService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME + "2");
		Collection services = findMultipleServices(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME + "2");
		assertEquals(1, services.size());

		for (Iterator iter = services.iterator(); iter.hasNext();) {
			Service s = (Service) iter.next();
			assertEquals(s.getName().getValue(), TEST_SERVICE_NAME + "2");
			assertEquals(s.getDescription().getValue(), TEST_SERVICE_DESC);
			assertEquals(2, s.getServiceBindings().size());
		}
	}

	/*
	 * testScenarioThree tests a scenario where there are two duplicate 
	 * services and one of the services has two EPRs and the other has none.
	 * The result should be a service with the two EPRs.
	 */
	public void testScenarioThree() throws ConfigurationException, JAXRException, RegistryException, URISyntaxException 
	{
		scenarioThree();
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		registry.findService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME + "3");
		Collection services = findMultipleServices(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME + "3");
		assertEquals(1, services.size());

		for (Iterator iter = services.iterator(); iter.hasNext();) {
			Service s = (Service) iter.next();
			assertEquals(s.getName().getValue(), TEST_SERVICE_NAME+"3");
			assertEquals(s.getDescription().getValue(), TEST_SERVICE_DESC);
			assertEquals(2, s.getServiceBindings().size());
		}
	}

	/*
	 * testScenarioFour tests a scenario one service has two EPRs and the 
	 * another service with a different name has none.    The result should
	 * be no change - two services, one with 2 eprs.
	 */
	public void testScenarioFour() throws ConfigurationException, JAXRException, RegistryException, URISyntaxException 
	{
		scenarioFour();
		JAXRRegistryImpl registry = new JAXRRegistryImpl();
		registry.findService(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME + "4");
		Collection services = findMultipleServices(TEST_SERVICE_CATEGORY, TEST_SERVICE_NAME + "4");
		assertEquals(1, services.size());
		
		boolean foundService = false;
		for (Iterator iter = services.iterator(); iter.hasNext();) {
			Service s = (Service) iter.next();
			if ((TEST_SERVICE_NAME+"4").equals(s.getName().getValue())) {
				assertEquals(s.getName().getValue(), TEST_SERVICE_NAME + "4");
				assertEquals(s.getDescription().getValue(), TEST_SERVICE_DESC);
				assertEquals(2, s.getServiceBindings().size());
				foundService = true;
			}
		}
		assertTrue(foundService);
	}

	/*
	 * Debugging method that is not used.
	 */
	public static void printService(Service service) throws JAXRException {
		logger.debug("Service name: " + service.getName().getValue());
		if (service.getDescription()!=null) {
			logger.debug("Description: " + service.getDescription().getValue());
		}
        for (Object c : service.getClassifications()) {
            Classification classific = (Classification ) c;                       
        	logger.debug("  Service Category: " + classific.getName().getValue());
        	logger.debug("  Service Description: " + classific.getDescription().getValue());
        	logger.debug("  Service Category: " + classific.getValue());
			logger.debug("  Number of EPRs : " + service.getServiceBindings().size());
        }
		logger.debug("Key id: " + service.getKey().getId());
	}
	
    public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(JAXRCoalesceUnitTest.class);
	}
}