/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.couriers;

import static org.junit.Assert.fail;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.jms.Destination;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.QueueConnection;
import javax.jms.QueueSession;
import javax.naming.Context;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.couriers.CourierTransportException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockejb.jms.MockQueue;
import org.mockejb.jms.QueueConnectionFactoryImpl;
import org.mockejb.jndi.MockContextFactory;

/**
 * JmsCourier unit tests.
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class JmsCourierUnitTest
{
    private static final String CONNECTION_FACTORY = "ConnectionFactory" ;
    private static final String QUEUE_NAME = "failQueue" ;
    
    private JMSEpr testEPR ;
    
    @Before
    public void setUp()
        throws Exception
    {
        MockContextFactory.setAsInitial();
        
        final Context ctx = NamingContextPool.getNamingContext(null);
        try
        {
            ctx.rebind(CONNECTION_FACTORY, new MockQueueConnectionFactory());
            ctx.rebind(QUEUE_NAME, new MockQueue(QUEUE_NAME));
        }
        finally
        {
            NamingContextPool.releaseNamingContext(ctx) ;
        }
        testEPR = new JMSEpr(JMSEpr.QUEUE_TYPE, QUEUE_NAME, CONNECTION_FACTORY) ;
        testEPR.getAddr().addExtension(Context.INITIAL_CONTEXT_FACTORY, System.getProperty(Context.INITIAL_CONTEXT_FACTORY)) ;
    }
    
    @After
    public void tearDown()
        throws Exception
    {
        MockContextFactory.revertSetAsInitial();
    }
    
    @Test(timeout=10000)
    public void testDelivery()
        throws Exception
    {
        final JmsCourier courier = new JmsCourier(testEPR) ;
        try
        {
            courier.deliver(MessageFactory.getInstance().getMessage()) ;
            fail("Expected to receive a CourierTransportException") ;
        }
        catch (final CourierTransportException cte) {} // expected
    }
    
    @Test(timeout=10000)
    public void testPickup()
        throws Exception
    {
        final JmsCourier courier = new JmsCourier(testEPR, true) ;
        try
        {
            courier.pickup(10) ;
            fail("Expected to receive a CourierTransportException") ;
        }
        catch (final CourierTransportException cte) {} // expected
    }
    
    private static final class MockQueueConnectionFactory extends QueueConnectionFactoryImpl
    {
        @Override
        public QueueConnection createQueueConnection() throws JMSException
        {
            return (QueueConnection)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {QueueConnection.class},
                    new MockQueueExceptionHandlerInvocationHandler(super.createQueueConnection())) ;
        }
    }
    
    private static final class MockQueueExceptionHandlerInvocationHandler implements InvocationHandler
    {
        private final QueueConnection queueConnection ;
        private ExceptionListener exceptionListener ;
            
        MockQueueExceptionHandlerInvocationHandler(final QueueConnection queueConnection)
        {
            this.queueConnection = queueConnection ;
        }
            
        public Object invoke(final Object proxy, final Method method, final Object[] args)
            throws Throwable
        {
            final String methodName = method.getName() ;
            if ("setExceptionListener".equals(methodName))
            {
                exceptionListener = (ExceptionListener)args[0] ;
                return null ;
            }
            else if ("getExceptionListener".equals(methodName))
            {
                return exceptionListener ;
            }
            else
            {
                final Object response = method.invoke(queueConnection, args) ;
                if (response instanceof QueueSession)
                {
                    final QueueSession queueSession = (QueueSession)response ;
                    return (QueueSession)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {QueueSession.class},
                            new MockQueueSessionInvocationHandler(queueSession)) ;
                }
                else
                {
                    return response ;
                }
            }
        }
    }
    
    private static final class MockQueueSessionInvocationHandler implements InvocationHandler
    {
        private final QueueSession queueSession ;
            
        MockQueueSessionInvocationHandler(final QueueSession queueSession)
        {
            this.queueSession = queueSession ;
        }
            
        public Object invoke(final Object proxy, final Method method, final Object[] args)
            throws Throwable
        {
            final String methodName = method.getName() ;
            if ("recover".equals(methodName))
            {
                return null ;
            }
            else if ("createConsumer".equals(methodName))
            {
                return new MockFailMessageConsumer() ;
            }
            else if ("createProducer".equals(methodName))
            {
                return new MockFailMessageProducer() ;
            }
            else
            {
                return method.invoke(queueSession, args) ;
            }
        }
    }
    
    private static final class MockFailMessageConsumer implements MessageConsumer
    {
        public void close() throws JMSException {}

        public MessageListener getMessageListener()
            throws JMSException
        {
            return null;
        }

        public String getMessageSelector()
            throws JMSException
        {
            return null;
        }

        public Message receive()
            throws JMSException
        {
            throw new JMSException("Deliberate receive exception") ;
        }

        public Message receive(long arg0)
        throws JMSException
        {
            throw new JMSException("Deliberate receive exception") ;
        }

        public Message receiveNoWait()
            throws JMSException
        {
            throw new JMSException("Deliberate receive exception") ;
        }

        public void setMessageListener(MessageListener arg0) throws JMSException {}
    }
    
    private static final class MockFailMessageProducer implements MessageProducer
    {
        public void close() throws JMSException {}

        public int getDeliveryMode()
            throws JMSException
        {
            return 0;
        }

        public Destination getDestination()
            throws JMSException
        {
            return null;
        }

        public boolean getDisableMessageID()
            throws JMSException
        {
            return false;
        }

        public boolean getDisableMessageTimestamp()
            throws JMSException
        {
            return false;
        }

        public int getPriority()
            throws JMSException
        {
            return 0;
        }

        public long getTimeToLive()
            throws JMSException
        {
            return 0;
        }

        public void send(Message arg0)
            throws JMSException
        {
            throw new JMSException("Deliberate send exception") ;
        }

        public void send(Destination arg0, Message arg1)
            throws JMSException
        {
            throw new JMSException("Deliberate send exception") ;
        }

        public void send(Message arg0, int arg1, int arg2, long arg3)
            throws JMSException
        {
            throw new JMSException("Deliberate send exception") ;
        }

        public void send(Destination arg0, Message arg1, int arg2, int arg3, long arg4)
            throws JMSException
        {
        }

        public void setDeliveryMode(int arg0) throws JMSException {}

        public void setDisableMessageID(boolean arg0) throws JMSException {}

        public void setDisableMessageTimestamp(boolean arg0) throws JMSException {}

        public void setPriority(int arg0) throws JMSException {}

        public void setTimeToLive(long arg0) throws JMSException {}
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(JmsCourierUnitTest.class);
    }
}
