/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;

import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.helpers.ConfigTree;

/**
 * Utilities class for managing JNDI related tasks.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class JndiUtil
{

    /**
     * General Namespace prefix for ESB related properties.  If the property name
     * is prefixed with this string it will be automatically added as one of the
     * JNDI prefixes.
     */
    public static final String ESB_JNDI_PARAMETER_PREFIX = "org.jboss.esb.";

    private JndiUtil()
    {
    }
    
    public static Set<String> getJndiPrefixesSet(final String jndiPrefixesValue)
    {
        final Set<String> jndiPrefixesSet = new TreeSet<String>();

        // Add default prefixes.
        jndiPrefixesSet.add("java.naming.") ;
        jndiPrefixesSet.add(ESB_JNDI_PARAMETER_PREFIX) ;

        if (jndiPrefixesValue != null)
        {
            final String[] values = jndiPrefixesValue.split(",") ;
            for (String value: values)
            {
                final String trim = value.trim() ;
                if (trim.length() > 0)
                {
                    jndiPrefixesSet.add(trim) ;
                }
            }
        }
        
        return jndiPrefixesSet;
    }
    
    /**
     * Get the list of JNDI extension prefixes.
     * 
     * @param jndiPrefixesValue The JNDI prefix list or null if default.
     * @return The list of JNDI extension prefixes.
     */
    public static String[] getJndiPrefixes(final String jndiPrefixesValue)
    {
        final Set<String> jndiPrefixesSet = getJndiPrefixesSet(jndiPrefixesValue);
        return jndiPrefixesSet.toArray(new String[jndiPrefixesSet.size()]) ;
    }
    
    /**
     * Parses the passed-in {@link ConfigTree} and extracts all properties that 
     * have been prefixed 
     * 
     * @param config
     * @return
     */
    public static Properties parseEnvironmentProperties(final ConfigTree config)
    {
        final Properties env = new Properties();
        
        // Comma separated list of prefixes. Will always include at least 'java.naming.'
        final String[] jndiPrefixes = getJndiPrefixes(getPropertyValue(config, JMSEpr.JNDI_PREFIXES));
        
        Set<String> attributeNames= config.getAttributeNames();
        for (final String name : attributeNames) 
        {
            for(String jndiPrefix: jndiPrefixes) 
            {
                if (name.startsWith(jndiPrefix))
                {
                    env.setProperty(name, config.getAttribute(name));
                    break;
                }
            }
        }
        
        return env;
    }
    
    public static String getPropertyValue(final ConfigTree config, final String propertyName)
    {
        return config.getAttribute(propertyName);
    }
    
    
}
