/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.StreamUtils;

/**
 * Util class for dealing with zip archives.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class JarUtil
{
    private JarUtil()
    {
    }
    
    /**
     * Extracts all the files ending with the fileSuffix and adds the file name as the key in 
     * the returned map. The contents of the file is the value of the map entry.
     * </p>
     * 
     * @param fileSuffix the suffix of files that should be returned.
     * @param charSet the character set that files should be read with.
     * @param inStream the ZipInputStream that is to be processed.
     * @return Map<String, String> A Map containg the file name as its key and the file contents as its value.
     * @throws IOException
     */
    public static Map<String, String> extractStringContents(final String fileSuffix, final String charSet, final ZipInputStream inStream) throws IOException
    {
        AssertArgument.isNotNullAndNotEmpty(fileSuffix, "fileSuffix");
        AssertArgument.isNotNullAndNotEmpty(charSet, "charSet");
        AssertArgument.isNotNull(inStream, "inStream");
        
        final Map<String, String> filesMap = new HashMap<String, String>();
        ZipEntry entry = null;
        while ((entry = inStream.getNextEntry()) != null)
        {
            final String entryName = entry.getName();
            if (entryName.endsWith(fileSuffix))
            {
                filesMap.put(entryName, StreamUtils.readStreamString(inStream, charSet));
            }
            else if (isJar(entryName))
            {
                final ByteArrayOutputStream bout = new ByteArrayOutputStream();
                byte[] byteReadBuffer = new byte[512];
                int byteReadCount;
                
                while((byteReadCount = inStream.read(byteReadBuffer)) != -1) 
                {
                    bout.write(byteReadBuffer, 0, byteReadCount);
                }
                final ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(bout.toByteArray());
                final ZipInputStream zipInputStream = new ZipInputStream(byteArrayInputStream);
                final Map<String, String> schemasFromNestedZip = extractStringContents(fileSuffix, zipInputStream);
                filesMap.putAll(schemasFromNestedZip);
            }
        }
        return filesMap;
    }
    
    /**
     * Extracts all the files ending with the fileSuffix and adds the file name as the key in 
     * the returned map. The contents of the file is the value of the map entry.
     * </p>
     * Calling this method is equivalent to calling:
     * extractStringContents(suffix, Charset.defaultCharset().name(), inStream).
     * 
     * @param fileSuffix the suffix of files that should be returned.
     * @param inStream the ZipInputStream that is to be processed.
     * @return Map<String, String> A Map containg the file name as its key and the file contents as its value.
     * @throws IOException
     */
    public static Map<String, String> extractStringContents(final String fileSuffix, final ZipInputStream inStream) throws IOException
    {
        return extractStringContents(fileSuffix, Charset.defaultCharset().name(), inStream);
    }
    
    private static boolean isJar(final String fileName)
    {
        return fileName.endsWith(".jar");
    }

}
