/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeaderElement;
import javax.xml.soap.SOAPMessage;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.auth.SecurityInfoExtractor;
import org.jboss.soa.esb.services.security.principals.User;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * WSSecurityInfoExtractor extracts security related information from
 * a SOAP security header. 
 * </p>
 * 
 * If the security element contains a 'UsernameToken' element this will be
 * extraced into a principal with the username and a credential being the
 * password represented as a character array. 
 * <br>
 * If the security element contains a 'BinarySecurityToken' element this informaiton
 * will be extracted into a credential which will be the content, or value, of
 * the BinarySecurityToken element.
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class WSSecuritySoapExtractor implements SecurityInfoExtractor<SOAPMessage>
{
	private Logger log = Logger.getLogger(WSSecuritySoapExtractor.class);
	
	/**
	 * Extracts UsernameToken element is one exists and creates
	 * a Principal with the username and sets a Credential using
	 * the password. The type of the Credential is a character array.
	 * 
	 * If the SOAP message contains a BinarySecurityToken this will be
	 * extracted and added as a Credential.
	 * <p>
	 * Note that this method i restrictive in reporting errors within the soap
	 * message. This is instead reflected in missing values in the returned
	 * AuthenticationRequest. This is done on purpose as not to give away
	 * any information about missing information. The SecuritySerivce impl
	 * should decide what information it wants a caller to recieve.
	 * 
	 * @param soap - the soap message represented as a String
	 * @return {@link AuthenticationRequest}
	 */
	public AuthenticationRequest extractSecurityInfo(final SOAPMessage soap)
	{
		if ( soap == null )
		{
			return null;
		}
		
		User user = null;
		Set<Object> credentials = new HashSet<Object>();
		
		try
		{
			final SOAPHeaderElement securityHeader = SOAPExtractorUtil.extractSecurityHeader( soap.getSOAPPart().getEnvelope() );
			if ( securityHeader == null )
			{
				return null;
			}	
			Iterator childElements = securityHeader.getChildElements();
			while ( childElements.hasNext() )
			{
    			final Node securityNode = (Node) childElements.next();
    			if ( securityNode.getNodeType() == Node.ELEMENT_NODE )
    			{
    				final String localName = securityNode.getLocalName();
        			if ( "BinarySecurityToken".equalsIgnoreCase( localName ) )
        			{
        				//	create a BinarySecurityToken (does some filtering and checking)
        				final BinarySecurityToken binaryToken = createBinarySecurityToken(securityNode);
        				// 	add the key(cert) as a credential
                		credentials.add(binaryToken.getKey());
        			}
        			else if ( "UsernameToken".equalsIgnoreCase(localName) )
        			{
        				UsernameToken usernameToken = createUsernameToken(securityNode);
        				final String userName = usernameToken.getUserName();
        				if ( userName != null )
        				{
            				user = new User(usernameToken.getUserName());
        				}	
        				char[] passwd = usernameToken.getPassword();
        				if ( passwd != null )
        				{
    						credentials.add(passwd);
        				}
        			}
    			}
			}
			// build the AuthenticationRequest. 
			if (user != null || !credentials.isEmpty())
	    		return new AuthenticationRequestImpl.Builder(user, credentials ).build();
			
		    return null;
		} 
		catch (final SOAPException e)
		{
			log.error("Caught a SOAPException while trying to extract security information: ", e);
			throw new SecurityException("Could not extract security info from SOAPMessage");
		}
	}
	
	private BinarySecurityToken createBinarySecurityToken( final Node node )
	{
		final NamedNodeMap attributes = node.getAttributes();;
		
		//	get the EncodingType
		final Node encodingTypeNode = attributes.getNamedItem("EncodingType");
		final String encodingType = encodingTypeNode.getNodeValue();
		
		//	get the ValueType
		final Node valueTypeNode = attributes.getNamedItem("ValueType");
		final String valueType = valueTypeNode.getNodeValue();
		
		//	get the certificate
		final String certString = node.getFirstChild().getNodeValue();
		
		//	create a BinarySecurityToken (does some filtering and checking for us
		final BinarySecurityToken binaryToken = new BinarySecurityToken();
		binaryToken.setEncodingType(encodingType);
		binaryToken.setValueType(valueType);
		binaryToken.setKey(certString);
		return binaryToken;
	}
	
	private UsernameToken createUsernameToken (final Node node)
	{
		final NodeList usernameElements = node.getChildNodes();
		int nrOfElements = usernameElements.getLength();
		final UsernameToken usernameToken = new UsernameToken();
		for ( int i = 0 ; i < nrOfElements ; i ++ )
		{
			final Node element = usernameElements.item(i);
			final String elLocalName = element.getLocalName();
			if ( "Username".equalsIgnoreCase( elLocalName ) )
			{
				usernameToken.setUserName(element.getFirstChild().getNodeValue());
			}
			else if ( "Password".equalsIgnoreCase( elLocalName ) )
			{
				usernameToken.setPassword(element.getFirstChild().getNodeValue());
			}
		}
		return usernameToken;
	}

}
