/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeaderElement;
import javax.xml.soap.SOAPMessage;

import org.apache.log4j.Logger;
import org.picketlink.identity.federation.core.wstrust.SamlCredential;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.auth.SecurityInfoExtractor;
import org.jboss.soa.esb.services.security.principals.User;
import org.w3c.dom.Element;
import org.w3c.dom.Node;


/**
 * Extracts SAML Assertions from a SOAPMessage. 
 * </p>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * 
 */
public class SamlSoapAssertionExtractor implements SecurityInfoExtractor<SOAPMessage>
{
    private Logger log = Logger.getLogger(SamlSoapAssertionExtractor.class);
    
    /**
     * Will extract a SAML security token from the passed in SOAP Message instance.
     * 
     * @param soap The SOAP message.
     * @return {@link AuthenticationRequest} A AuthenticationRequest containing a SamlCredential, or null if no SAML token was present.
     */
    public AuthenticationRequest extractSecurityInfo(final SOAPMessage soap)
    {
        if ( soap == null )
        {
            return null;
        }
        
        User user = null;
        Set<Object> credentials = new HashSet<Object>();
        
        try
        {
            final SOAPHeaderElement securityHeader = SOAPExtractorUtil.extractSecurityHeader( soap.getSOAPPart().getEnvelope() );
            if ( securityHeader == null )
            {
                return null;
            }   
            Iterator childElements = securityHeader.getChildElements();
            while (childElements.hasNext())
            {
                final Node securityNode = (Node) childElements.next();
                if ( securityNode.getNodeType() == Node.ELEMENT_NODE )
                {
                    final String localName = securityNode.getLocalName();
                    if ("Assertion".equalsIgnoreCase(localName))
                    {
                        credentials.add(new SamlCredential((Element)securityNode));
                    }
                }
            }
            
            // build the AuthenticationRequest. 
            return new AuthenticationRequestImpl.Builder(user, credentials ).build();
        } 
        catch (final SOAPException e)
        {
            log.error("Caught a SOAPException while trying to extract security information: ", e);
            throw new SecurityException("Could not extract security info from SOAPMessage");
        }
    }

}
