/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import java.io.ByteArrayInputStream;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;

import org.apache.commons.codec.binary.Base64;

/**
 * Represents a WS-Security BinarySecurityToken.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class BinarySecurityToken
{
	private String encodingType;
	private String valueType;
	private Certificate cert;
	private enum EncodingType { Base64Binary, HexBinary }

	public String getEncodingType()
	{
		return encodingType;
	}

	public void setEncodingType(String encodingType)
	{
		this.encodingType = stripNS(encodingType);
	}

	public String getValueType()
	{
		return valueType;
	}

	public void setValueType(String valueType)
	{
		this.valueType = stripNS(valueType);
	}

	public Certificate getKey()
	{
		return cert;
	}

	public void setKey(final String key)
	{
		try
		{
			byte[] keyBytes = null;
			if ( encodingType.equalsIgnoreCase( EncodingType.Base64Binary.toString() ) )
			{
			    Base64 decoder = new Base64();
    			keyBytes = decoder.decode(key.getBytes());
			}
			else
			{
				keyBytes = key.getBytes();
			}

			CertificateFactory factory = CertificateFactory.getInstance( certificateMatch( valueType ) );
			cert = factory.generateCertificate( new ByteArrayInputStream( keyBytes ));
		}
		catch (CertificateException e)
		{
			throw new IllegalStateException("Could not create certificate: ", e);
		}
	}

	private String certificateMatch(final String valueType)
	{
		if ( valueType.startsWith("X509") )
			return "X.509";

		return valueType;
	}

	private String stripNS(String value)
	{
		if ( value != null )
		{
			if ( value.startsWith("http"))
			{
				final int idx = value.indexOf('#');
    			if ( idx > 0 )
    				value = value.substring( idx + 1 );
			}
			else
			{
    			final int idx = value.indexOf(':');
    			if ( idx > 0 )
    				value = value.substring( idx + 1 );
			}
		}
		return value;
	}

}
