/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.parameters;

import org.jboss.internal.soa.esb.parameters.ParamFileRepository;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Factory class for Singleton {@link ParamRepository} creation. <p/> Uses the
 * "org.jboss.soa.esb.paramsRepository.class" system property to determine the
 * {@link ParamRepository} implementation to be created. If not specified, the
 * default implementation is the
 * {@link org.jboss.internal.soa.esb.parameters.ParamFileRepository}.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class ParamRepositoryFactory
{

	/**
	 * Singleton instance of the repository.
	 */
	private static ParamRepository instance;

	/**
	 * Factory method for creating the Singleton {@link ParamRepository}
	 * instance for the system.
	 * 
	 * @return The ParamRepository instance for the system.
	 */
	public static ParamRepository getInstance()
	{
		if (instance == null)
		{
			// Hasn't been created yet - lets created it in a synchronized
			// fashion...
			instance = synchedGetInstance();
		}

		return instance;
	}

	/**
	 * Synchronizes creation of the Singleton instance.
	 * 
	 * @return The Singleton ParamRepository instance.
	 */
	private static ParamRepository synchedGetInstance()
	{
		synchronized (ParamRepositoryFactory.class)
		{
			// The following check ensures that multithreads won't create
			// multiple instances of the singleton. Threads that are blocked
			// waiting
			// at the start of this synch block will simply fall through and
			// return the
			// instance created by the thread that got in ahead of them.
			if (instance == null)
			{
				String runtimeClassName = Configuration
						.getParamRepositoryImplClass();

				if (runtimeClassName == null)
				{
					// If there's no repository name configured, return the
					// default file-based repository...
					instance = new ParamFileRepository();
				} else
				{
					try
					{
						Class<?> runtimeClass = ClassUtil.forName(runtimeClassName, ParamRepositoryFactory.class);
						instance = (ParamRepository) runtimeClass.newInstance();
					} catch (Exception e)
					{
						throw new IllegalStateException(
								"System Configuration Exception: Unable to create system "
										+ ParamRepository.class.getSimpleName()
										+ " instance from runtime class name ["
										+ runtimeClassName + "]", e);
					}
				}
			}
		}

		return instance;
	}

	/**
	 * Resets the factory, allowing creation of a new Singleton instance. <p/>
	 * Should only need to be used for testing purposes.
	 */
	protected static void reset()
	{
		synchronized (ParamRepositoryFactory.class)
		{
			instance = null;
		}
	}
}
