/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message.mime;

import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;

/**
 * Mime decoder for mime type "text/plain".
 * <p/>
 * Decodes the bytes to a simple String.
 * 
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
@MimeType("text/plain")
public class TextPlainMimeDecoder implements MimeDecoder, Configurable {

	private Charset encodingCharset;
	private String decodeTo;

	public void setConfiguration(ConfigTree configTree) throws ConfigurationException {
		AssertArgument.isNotNull(configTree, "configTree");
		
		// Encoding config...
		String encoding = configTree.getAttribute("encoding", "UTF-8");
		encodingCharset = Charset.forName(encoding);
				
		// Decode To config...
		decodeTo = configTree.getAttribute("decodeTo", "STRING");
	}

	public Object decode(byte[] bytes) throws MimeDecodeException { 
		if(decodeTo.equals("BYTES")) {
			return bytes;
		}
		
		try {
			String decodedPayload = new String(bytes, encodingCharset.name());
			
			if(decodeTo.equals("CHARS")) {
				return decodedPayload.toCharArray();
			} else {
				return decodedPayload;
			}
		} catch (UnsupportedEncodingException e) {
			throw new MimeDecodeException("Unexpected character encoding error.", e);
		}
	}
}
