/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message.mime;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Mime handler.
 * 
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
public interface MimeDecoder {
	
	/**
	 * Decode the specified array of bytes based on the {@link MimeType}
	 * annotation on the implementing {@link MimeDecoder}.
	 * @param bytes The bytes to be decoded.
	 * @return The decoded bytes.
	 * @throws MimeDecodeException Error while decoding the byts array.
	 */
	Object decode(byte[] bytes) throws MimeDecodeException;
	
	// TODO: Add support for encoding to a byte[] ?
	
	/**
	 * Factory class.
	 */
	public static class Factory {
		
		public static final String MIME_TYPE = "mimeType";
		public static final String MIME_DECODER = "mimeDecoder";
		public static final String DECODERS_LST_RESOURCE = "META-INF/org/jboss/soa/esb/listeners/message/mime/decoders.lst";
		
	    private final static Logger logger = Logger.getLogger(MimeDecoder.class) ;
		
	    /**
	     * Get a MimeDecoder instance by mime type.
	     * @param mimeType The mime type.
	     * @return The MimeDecoder for that mime type.
	     * @throws ConfigurationException Error creating MimeDecoder instance.
	     */
		public static MimeDecoder getInstanceByType(String mimeType) throws ConfigurationException {
			AssertArgument.isNotNullAndNotEmpty(mimeType, MIME_TYPE);
			mimeType = mimeType.trim();
			
			List<URL> decoderListFiles;
			
			try {
				decoderListFiles = ClassUtil.getResources(DECODERS_LST_RESOURCE, MimeDecoder.class);
			} catch (IOException e) {
				throw new ConfigurationException("Unexpected error locating mime decoder.lst files on classpath (" + DECODERS_LST_RESOURCE + ").", e);
			}
			
			for(URL decoderListFile : decoderListFiles) {
				try {
					InputStream listFileStream = decoderListFile.openStream();
					try {
						BufferedReader reader = new BufferedReader(new InputStreamReader(listFileStream, "UTF-8"));
						String line;
						
						while((line = reader.readLine()) != null) {
							line = line.trim();
							if(line.length() > 0 && !line.startsWith("#")) {
								Class<?> decoderClass;
								try {
									decoderClass = ClassUtil.forName(line, MimeDecoder.class);
								} catch (ClassNotFoundException e) {
									logger.error("MimeDecoder class '" + line + "' (listed in decoder.lst file '" + decoderListFile + "') is not available on the classpath.", e);
									continue;
								}

								if(MimeDecoder.class.isAssignableFrom(decoderClass)) {
									MimeType mimeTypeAnno = decoderClass.getAnnotation(MimeType.class);
									if(mimeTypeAnno == null) {
										logger.error("MimeDecoder class '" + line + "' (listed in decoder.lst file '" + decoderListFile + "') is not annotated with the MimeType annotation.");
										continue;
									}
									if(mimeTypeAnno.value().trim().equals(mimeType)) {
										// Found the MimeDecoder...
										try {
											return (MimeDecoder) decoderClass.newInstance();
										} catch (Exception e) {
											throw new ConfigurationException("Found MimeDecoder class '" + line + "' (listed in decoder.lst file '" + decoderListFile + "') but was unable to instanciate it.  Must have a public default constructor.", e);
										}
									}
								} else {
									logger.error("MimeDecoder class '" + line + "' (listed in decoder.lst file '" + decoderListFile + "') doesn't implement the MimeDecoder interface.");
									continue;
								}								
							}
						}
					} finally {
						listFileStream.close();
					}
				} catch (IOException e) {
					throw new ConfigurationException("Unexpected error reading mime decoder.lst files on classpath (" + decoderListFile + ").", e);
				}
			}			

			throw new ConfigurationException("Failed to find a MimeDecoder implementation for mime type '" + mimeType + "'.  MimeDecoders must be listed in a '" + DECODERS_LST_RESOURCE + "' file on the classpath.");
		}
		
	    /**
	     * Get a MimeDecoder instance by class name.
	     * @param className The MimeDecoder class name.
	     * @return The MimeDecoder instance.
	     * @throws ConfigurationException Error creating MimeDecoder instance.
	     */
		public static MimeDecoder getInstanceByClassName(String className) throws ConfigurationException {
			AssertArgument.isNotNullAndNotEmpty(className, "className");
			className = className.trim();
			
			Class<?> decoderClass;
			try {
				decoderClass = ClassUtil.forName(className, MimeDecoder.class);
			} catch (ClassNotFoundException e) {
				throw new ConfigurationException("MimeDecoder class '" + className + "' is not available on the classpath.", e);
			}

			if(MimeDecoder.class.isAssignableFrom(decoderClass)) {
				try {
					return (MimeDecoder) decoderClass.newInstance();
				} catch (Exception e) {
					throw new ConfigurationException("Unable to instanciate MimeDecoder class '" + className + "'.  Must have a public default constructor.", e);
				}
			} else {
				throw new ConfigurationException("MimeDecoder class '" + className + "' doesn't implement the MimeDecoder interface.");
			}
		}

	    /**
	     * Get a MimeDecoder instance by {@link ConfigTree}.
	     * @param config The ConfigTree instance.
	     * @return The MimeDecoder instance.
	     * @throws ConfigurationException Error creating MimeDecoder instance.
	     */
		public static MimeDecoder getInstanceByConfigTree(ConfigTree config) throws ConfigurationException {
			AssertArgument.isNotNull(config, "config");
			MimeDecoder mimeDecoder;
			
			String mimeType = config.getAttribute(MIME_TYPE);		
			if(mimeType != null) {
				mimeDecoder = getInstanceByType(mimeType);
			} else {
				String mimeDecoderClass = config.getAttribute(MIME_DECODER);
				if(mimeDecoderClass != null) {
					mimeDecoder = getInstanceByClassName(mimeDecoderClass);
				} else {
					mimeDecoder = new NullMimeDecoder();
				}
			}
		
			if(mimeDecoder instanceof Configurable) {
				((Configurable)mimeDecoder).setConfiguration(config);
			}
			
			return mimeDecoder;
		}
	}
}
