/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.PickUpOnlyCourier;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.util.DefaultReplyTo;
import org.jboss.soa.esb.couriers.Courier;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;


/**
 * 
 * Utility class to hide implementation details for sending Command messages and optionally awaiting for a response. 
 * 
 * @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a> 
 * @deprecated Use {@link org.jboss.soa.esb.client.ServiceInvoker}.
 */

public class Invoker {

	/**
	 * 
	 * @param message Message - the message to deliver
	 * @param category String - for Registry search
	 * @param name String - for Registry search
	 * @throws RegistryException
	 * @throws MalformedEPRException
	 * @throws CourierException
	 */
	public static void invoke(Message message, String category, String name)
		throws RegistryException, MalformedEPRException, CourierException, ServiceNotFoundException
	{
		try { invokeAndAwaitResponse(message,category,name,-1); }
		catch (CourierTimeoutException e)
		{
			_logger.fatal("Unexpected CourierTimeoutException caught!");
		}
	} //________________________________
	
	/**
	 * Encapsulate command in an ESB Message, deliverAsync it, and wait for a response Message.
	 * 
	 * @param message Message - the message to deliver
	 * @param category String - Service category name for Registry inquiry
	 * @param name String - Service name for Registry inquiry
	 * @param maxWaitMillis int - Maximum time to wait for a response
	 * @return Message
	 * @throws RegistryException
	 * @throws MalformedEPRException
	 * @throws CourierException
	 * @throws CourierTimeoutException - If response was not received in specified time
	 */
	
	public static Message invokeAndAwaitResponse(Message message, String category, String name
						,int maxWaitMillis)
		throws RegistryException, MalformedEPRException, CourierException, CourierTimeoutException, ServiceNotFoundException
	{
		Call call	= message.getHeader().getCall();
		Collection<EPR> eprs = RegistryUtil.getEprs(category, name);
		if (null==eprs || eprs.size()<1)
			throw new RegistryException("No eprs found for <"+category+","+name+">");
		EPR service = eprs.iterator().next();
		call.setTo(service);
		return invokeAndAwaitResponse(message, maxWaitMillis);
	}
	
	/**
	 * Deliver an ESB Message and wait for a response Message.
	 * 
	 * @param outgoing Message - should contain at least the toEPR
	 * @param maxWaitMillis int - Maximum time to wait for a response
	 * @return Message
	 * @throws CourierException
	 * @throws MalformedEPRException
	 * @throws CourierTimeoutException - If response was not received in specified time
	 */
	public static Message invokeAndAwaitResponse(Message outgoing ,int maxWaitMillis)
		throws CourierException, MalformedEPRException, CourierTimeoutException
	{
		Call call	= outgoing.getHeader().getCall();
		if (null==call.getMessageID())
		{
			URI	 uri	= null;
			try  { uri = new URI(UUID.randomUUID().toString()); }
			catch (URISyntaxException e)
			{
				_logger.fatal("Failed to create URI: "+e);				
				throw new CourierException(e);
			}
			call.setMessageID(uri);
		}
		
		Courier sender = CourierFactory.getCourier(call.getTo());
                PickUpOnlyCourier receiver = null;
                try
                {
                        boolean waitForResponse = (maxWaitMillis > 0);
                        if (waitForResponse)
                        {
                                if (null==call.getReplyTo())
                                        call.setReplyTo(DefaultReplyTo.getReplyTo(call.getTo()));
                                receiver        = CourierFactory.getPickupCourier(call.getReplyTo());
                        }
                        
                        sender.deliver(outgoing);
                        return (waitForResponse ? receiver.pickup(maxWaitMillis) : null) ;
                }
                finally
                {
                    CourierUtil.cleanCourier(sender) ;
                    CourierUtil.cleanCourier(receiver) ;
                }
	} //________________________________
	
	private static Logger _logger = Logger.getLogger(Invoker.class);
}
