/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.jca;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.resource.spi.ActivationSpec;
import javax.resource.spi.endpoint.MessageEndpointFactory;

import org.apache.log4j.Logger;
import org.jboss.metadata.ActivationConfigPropertyMetaData;
import org.jboss.mx.util.JMXExceptionDecoder;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * interfaces with JCA to activate an endpoint
 *
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @version $Revision: 1.1 $
 */
public class JBoss42ActivationBridge implements ActivationBridge
{
   private static Logger logger = Logger.getLogger(JBoss42ActivationBridge.class);

   /**
    * The signature for createActivationSpec
    */
   private static final String[] createActivationSpecSig = new String[]
           {
                   Class.class.getName(),
                   Collection.class.getName()
           };

   /**
    * The signature for activate/deactivateEndpint
    */
   private static final String[] activationSig = new String[]
           {
                   MessageEndpointFactory.class.getName(),
                   ActivationSpec.class.getName()
           };

   private String adapter;
   private Class messagingTypeClass;
   private MessageEndpointFactory messageEndpointFactory;
   private Map<String, String> activationProperties;


   private ActivationSpec activationSpec;
   private MBeanServer mbeanServer;
   private ObjectName resourceAdapterObjectName;


   public void setAdapter(String adapter)
   {
      this.adapter = adapter;
   }

   public void setMessagingTypeClass(Class messagingTypeClass)
   {
      this.messagingTypeClass = messagingTypeClass;
   }

   public void setMessageEndpointFactory(MessageEndpointFactory messageEndpointFactory)
   {
      this.messageEndpointFactory = messageEndpointFactory;
   }

   public void setActivationProperties(Map<String, String> activationProperties)
   {
      this.activationProperties = activationProperties;
   }

   public void activate()
   {
      mbeanServer = MBeanServerLocator.locateJBoss();
      try
      {
         resourceAdapterObjectName = new ObjectName("jboss.jca:service=RARDeployment,name='" + adapter + "'");
      }
      catch (MalformedObjectNameException e)
      {
         throw new RuntimeException("Unable to create JMX ObjectName to register adapter: " + adapter, e);
      }

      ArrayList<ActivationConfigPropertyMetaData> metaData = new ArrayList<ActivationConfigPropertyMetaData>();
      for (String key : activationProperties.keySet())
      {
         String value = activationProperties.get(key);
         metaData.add(createActivationConfigPropertyMetaData(key, value));
         //metaData.add(new ActivationConfigPropertyMetaData(key, value));
      }

      Object[] params = new Object[]
              {
                      messagingTypeClass,
                      metaData
              };
      try
      {
         activationSpec = (ActivationSpec) mbeanServer.invoke(resourceAdapterObjectName, "createActivationSpec", params, createActivationSpecSig);
      }
      catch (Throwable t)
      {
         t = JMXExceptionDecoder.decode(t);
         throw new RuntimeException("Endpoint activation failed ra=" + resourceAdapterObjectName +
                 " activationSpec=" + activationSpec, t);
      }

      params = new Object[]{messageEndpointFactory, activationSpec};
      try
      {
         mbeanServer.invoke(resourceAdapterObjectName, "endpointActivation", params, activationSig);
      }
      catch (Throwable t)
      {
         t = JMXExceptionDecoder.decode(t);
         throw new RuntimeException("Endpoint activation failed ra=" + resourceAdapterObjectName +
                 " activationSpec=" + activationSpec, t);
      }
   }
   
   private ActivationConfigPropertyMetaData createActivationConfigPropertyMetaData(final String key, final String value) 
    {
        try
        {
            Class<?> c = ClassUtil.forName("org.jboss.metadata.ejb.spec.ActivationConfigPropertyMetaData" , getClass());
            Object instance = c.newInstance();
            // Call setActivationConfigPropertyName with the key.
            Method setName = c.getMethod("setActivationConfigPropertyName", new Class[] { String.class});
            setName.invoke(instance, key);
            // Call setValue with the value.
            Method setValue = c.getMethod("setValue", String.class);
            setValue.invoke(instance, value);
            
            // Now create the AS 5 version of ActivationConfigPropertyMetaData.
            Class<?> returnClass = ClassUtil.forName("org.jboss.metadata.ActivationConfigPropertyMetaData" , getClass());
            // Call the constructor using the above created instance as the argument.
            Constructor<?> constructor = returnClass.getConstructor(c);
            return (ActivationConfigPropertyMetaData) constructor.newInstance(instance);
        } 
        catch (final ClassNotFoundException e)
        {
            return new ActivationConfigPropertyMetaData(key, value);
        } 
        catch (final InstantiationException e)
        {
            throw new RuntimeException("Could not instantiate an instance of org.jboss.metadata.ejb.spec.ActivationConfigPropertyMetaData", e);
        } 
        catch (final IllegalAccessException e)
        {
            throw new RuntimeException("IllegalAccessException while trying to create an instance of org.jboss.metadata.ejb.spec.ActivationConfigPropertyMetaData", e);
        } 
        catch (SecurityException e)
        {
            throw new RuntimeException(e.getMessage(), e);
        } 
        catch (NoSuchMethodException e)
        {
            throw new RuntimeException(e.getMessage(), e);
        } 
        catch (IllegalArgumentException e)
        {
            throw new RuntimeException(e.getMessage(), e);
        } 
        catch (InvocationTargetException e)
        {
            throw new RuntimeException(e.getMessage(), e);
        } 
   }

   /**
    * Deactivate
    */
   public void deactivate()
   {
      Object[] params = new Object[]{messageEndpointFactory, activationSpec};
      try
      {
         mbeanServer.invoke(resourceAdapterObjectName, "endpointDeactivation", params, activationSig);
      }
      catch (Throwable t)
      {
         t = JMXExceptionDecoder.decode(t);
         logger.warn("Endpoint activation failed ra=" + resourceAdapterObjectName +
                 " activationSpec=" + activationSpec, t);
      }
   }

}
