/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.Session;
import javax.jms.Topic;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsSession;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.filter.FilterManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.listeners.lifecycle.AbstractThreadedManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleThreadState;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.jboss.soa.esb.services.security.PasswordUtil;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.JmsUtil;
import org.jboss.soa.esb.util.JndiUtil;

public class JmsGatewayListener extends AbstractThreadedManagedLifecycle {

    private String durableSubscriptionName;

	public JmsGatewayListener(ConfigTree listenerConfig)
            throws ConfigurationException {
        super(listenerConfig);
        _config = listenerConfig;
        checkMyParms();
    } // __________________________________

    /**
     * Handle the initialisation of the managed instance.
     *
     * @throws ManagedLifecycleException for errors while initialisation.
     */
    protected void doInitialise() throws ManagedLifecycleException {
    	// Needed to retain prior semantics of fail-on-initialise if service lookup fails
        try {
        	Collection<EPR> _targetEprs = RegistryUtil.getEprs(_targetServiceCategory,
                    _targetServiceName);
            if (null == _targetEprs || _targetEprs.size() < 1)
                throw new ManagedLifecycleException("EPR <"
                        + _targetServiceName + "> not found in registry");
        } catch (ServiceNotFoundException snfe) {
        	throw new ManagedLifecycleException("EPR <" + _targetServiceName + " "
            + _targetServiceName + "> not found in registry");
       }
        catch (final RegistryException re) {
            throw new ManagedLifecycleException(
                    "Unexpected registry exception", re);
        }

        try {
        	_serviceInvoker = new ServiceInvoker(_targetServiceCategory, _targetServiceName);
        } catch (MessageDeliverException e) {
        	throw new ManagedLifecycleException(e);
        }

        durableSubscriptionName = _config.getAttribute(JMSEpr.DURABLE_SUBSCRIPTION_NAME);
        
        try {
            prepareMessageReceiver();
        }
        catch (final ConnectionException ce) {
            throw new ManagedLifecycleException(
                    "Unexpected connection exception from prepareMessageReceiver",
                    ce);
        }
        catch (final JMSException jmse) {
            throw new ManagedLifecycleException(
                    "Unexpected JMS error from prepareMessageReceiver", jmse);
        }
        catch (final ConfigurationException ce) {
            throw new ManagedLifecycleException(
                    "Unexpected configuration exception from prepareMessageReceiver",
                    ce);
        } catch (final NamingContextException nce) {
            throw new ManagedLifecycleException(
                    "Unexpected naming context exception from prepareMessageReceiver",
                    nce);
        }

        if (_serviceName != null) {
            try {
                RegistryUtil.register(_config, _myEpr);
            }
            catch (final RegistryException re) {
                throw new ManagedLifecycleException(
                        "Unexpected error during registration for epr "
                                + _myEpr, re);
            }
        }
    }

    /**
     * Execute on the thread.
     */
    protected void doRun() {
        if (_logger.isDebugEnabled()) {
            _logger.debug("run() method of " + this.getClass().getSimpleName()
                    + " started on thread " + Thread.currentThread().getName());
        }

        while (isRunning()) {
            javax.jms.Message msgIn = receiveOne();
            
            if (null != msgIn) {
                try {
                    Object obj = _processMethod.invoke(_composer, new Object[] {msgIn});
                    
                    if (null == obj) {
                        _logger.warn("Action class method <"
                                + _processMethod.getName()
                                + "> returned a null object");
                    } else {
                        // try to deliverAsync the composed message, using the
                        // appropriate courier
                        // to the target service

                        Map<String, Object> params = new HashMap<String, Object>();

                        params.put(Environment.GATEWAY_CONFIG, _config);

                        obj = FilterManager.getInstance().doOutputWork((Message) obj, params);

                        try {
                        	Message message = (Message) obj;
                        	_serviceInvoker.deliverAsync(message);
                        }
                        catch (ClassCastException e) {
                            _logger.error("Action class method <"
                                    + _processMethod.getName()
                                    + "> returned a non Message object", e);
                            
                            rollbackJMSTransaction();
                        }
                    }
                    if (jmsSession.getTransacted()) {
                        jmsSession.commit() ;
                    } else {
                        msgIn.acknowledge() ;
                    }
                }
                catch (InvocationTargetException e) {
                    _logger.error("Problems invoking method <"
                            + _processMethod.getName() + ">", e);
                    rollbackJMSTransaction();
                }
                catch (IllegalAccessException e) {
                    _logger.error("Problems invoking method <"
                            + _processMethod.getName() + ">", e);
                    rollbackJMSTransaction();
                }
                catch (Exception e) {
                    _logger.error("Unexpected problem", e);
                    rollbackJMSTransaction();
                }
            }
        }

        _logger.debug("run() method of " + this.getClass().getSimpleName()
                + " finished on thread " + Thread.currentThread().getName());
    } // ________________________________

    /**
     * Handle the threaded destroy of the managed instance.
     *
     * @throws ManagedLifecycleException for errors while destroying.
     */
    protected void doThreadedDestroy() throws ManagedLifecycleException {
        cleanup();
    }
    
    private void rollbackJMSTransaction() 
    {
        try {
            if (jmsSession.getTransacted()) {
                jmsSession.rollback() ;
            } else {
                releaseSession() ;
            }
        } catch (final JMSException jmse) {
            releaseSession() ;
        }
    }
    

    private void releaseSession() {
        if (jmsSession != null) {
            try {
                jmsConnectionPool.releaseSession(jmsSession);
            } finally {
                jmsSession = null;
            }
        }
    }

    private void cleanup() {
        try {
            if (_serviceName != null) {
                RegistryUtil.unregister(_serviceCategory, _serviceName, _myEpr);
            }
            if ( jmsSession != null && jmsSession.getTransacted() )
            	jmsSession.rollback();
        } 
        catch (JMSException e)
		{
			e.printStackTrace();
		} finally {
            try {
                if (jmsMessageConsumer != null) {
                    try {
                        jmsMessageConsumer.close();
                    }
                    catch (final JMSException jmse) {
                    } // ignore
                }
            } finally {
                if (jmsSession != null) {
                    jmsConnectionPool.closeSession(jmsSession);
                }
            }
        }
    }

    /**
     * Check for mandatory and optional attributes in parameter tree
     *
     * @throws ConfigurationException -
     *                                if mandatory atts are not right or actionClass not in
     *                                classpath
     */
    protected void checkMyParms() throws ConfigurationException {

        _targetServiceCategory = ListenerUtil.getValue(_config,
                ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, null);
        _targetServiceName = ListenerUtil.getValue(_config,
                ListenerTagNames.TARGET_SERVICE_NAME_TAG, null);

        if (_targetServiceCategory == null)
        	throw new ConfigurationException("No service category defined!");
        
        if (_targetServiceName == null)
        	throw new ConfigurationException("No service name defined!");
        
        jmsDestinationName = ListenerUtil.getValue(_config,
                JMSEpr.DESTINATION_NAME_TAG, null);

        if (jmsDestinationName == null)
        	throw new ConfigurationException("No queue name defined!");
        
        resolveComposerClass();

        // No problem if selector is null - everything in queue will be returned
        _messageSelector = _config.getAttribute(JMSEpr.MESSAGE_SELECTOR_TAG);
        _logger.debug("No value specified for: " + JMSEpr.MESSAGE_SELECTOR_TAG
                + " - All messages in queue will be received by this listener");
    } // ________________________________

    protected void resolveComposerClass() throws ConfigurationException {
        try {
            String sProcessMethod = null;
            _composerName = _config
                    .getAttribute(ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG);
            if (null != _composerName) { // class attribute
                _composerClass = ClassUtil.forName(_composerName, getClass());
                Constructor oConst = _composerClass.getConstructor(new Class[]
                        {ConfigTree.class});
                _composer = oConst.newInstance(_config);
                sProcessMethod = _config
                        .getAttribute(
                                ListenerTagNames.GATEWAY_COMPOSER_METHOD_TAG,
                                "process");
            } else {
                _composerName = PackageJmsMessageContents.class.getName();
                _composerClass = PackageJmsMessageContents.class;
                _composer = new PackageJmsMessageContents(_config);
                sProcessMethod = "process";
                _logger
                        .debug("No <" + ListenerTagNames.ACTION_ELEMENT_TAG
                                + "> element found in configuration"
                                + " -  Using default composer class : "
                                + _composerName);
            }

            _processMethod = _composerClass.getMethod(sProcessMethod,
                    new Class[]
                            {Object.class});
        }
        catch (Exception ex) {
            throw new ConfigurationException(ex);
        }
    } // ________________________________

    private void prepareMessageReceiver() throws ConfigurationException,
            JMSException, ConnectionException, NamingContextException {
        jmsSession = null;
        jmsDestination = null;

        Properties environment = new Properties();

        _config.mapTo(environment, JMSEpr.JNDI_URL_TAG, Context.PROVIDER_URL);
        _config.mapTo(environment, JMSEpr.JNDI_CONTEXT_FACTORY_TAG, Context.INITIAL_CONTEXT_FACTORY);
        _config.mapTo(environment, JMSEpr.JNDI_PKG_PREFIX_TAG, Context.URL_PKG_PREFIXES);
        _config.mapTo(environment, JMSEpr.MAX_SESSIONS_PER_CONNECTION);
        _config.mapTo(environment, JMSEpr.MAX_XA_SESSIONS_PER_CONNECTION);
        _config.mapTo(environment, JMSEpr.CLIENT_ID);

        Set<String> names = _config.getAttributeNames();
        final String jndiPrefixesValue = _config.getAttribute(JMSEpr.JNDI_PREFIXES) ;
        if (jndiPrefixesValue != null) {
            environment.setProperty(JMSEpr.JNDI_PREFIXES, jndiPrefixesValue) ;
        }
        final String[] jndiPrefixes = JndiUtil.getJndiPrefixes(jndiPrefixesValue) ;
        for (String name : names) {
            for(String jndiPrefix: jndiPrefixes) {
                if (name.startsWith(jndiPrefix)) {
                    environment.setProperty(name, _config.getAttribute(name));
                    break ;
                }
            }
        }
        Context oJndiCtx = NamingContextPool.getNamingContext(environment);
        try {
            String sFactClass = ListenerUtil.getValue(_config,
                    JMSEpr.CONNECTION_FACTORY_TAG, "ConnectionFactory");
            if (null == _config.getAttribute(JMSEpr.CONNECTION_FACTORY_TAG))
                _logger.debug("No value specified for "
                        + JMSEpr.CONNECTION_FACTORY_TAG + " attribute"
                        + " -  Using default of: '" + sFactClass + "'");
            _serviceCategory = _config
                    .getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG);
            _serviceName = _config.getAttribute(ListenerTagNames.SERVICE_NAME_TAG);
    
            String destType = _config.getAttribute(JMSEpr.DESTINATION_TYPE_TAG);
            boolean persistent = Boolean.valueOf( _config.getAttribute(JMSEpr.PERSISTENT_TAG));
            boolean transacted = Boolean.valueOf( _config.getAttribute(JMSEpr.TRANSACTED_TAG));
            _logger.debug( "JMSGateway isTransacted = " + transacted );
            
            String acknowledgeMode = _config.getAttribute(JMSEpr.ACKNOWLEDGE_MODE_TAG);
            
            final String username =  _config.getAttribute( JMSEpr.JMS_SECURITY_PRINCIPAL_TAG );
            
            // password can be either a clear text password or a file containting an encrypted password.
            final String password =  _config.getAttribute( JMSEpr.JMS_SECURITY_CREDENTIAL_TAG );
            String decryptedPassword = null;
            if (JmsUtil.isSecurityConfigured(username, password))
            {
    	        environment.put( JMSEpr.JMS_SECURITY_PRINCIPAL_TAG, username );
    	        decryptedPassword = JmsUtil.getPasswordFromFile(password);
    	        environment.put( JMSEpr.JMS_SECURITY_CREDENTIAL_TAG, decryptedPassword);
            }
            
            // When creating the EPR we always use the password as seen in the configuration.
            _myEpr = (null == _serviceName) ? null : new JMSEpr(JMSEpr.ONE_ONE_PROTOCOL, destType,
                jmsDestinationName, sFactClass, environment, _messageSelector, persistent, acknowledgeMode,
                username, password, transacted );
            
            // To create the connection pool we need to use the decrypted password (if applicable).
            jmsConnectionPool = JmsConnectionPoolContainer.getPool(environment, sFactClass, username, decryptedPassword);
            	
            try {
                jmsSession = _myEpr != null ? jmsConnectionPool.getSession(((JMSEpr)_myEpr).getAcknowledgeMode()):
                	jmsConnectionPool.getSession(Session.AUTO_ACKNOWLEDGE);
            		
            }
            catch (NamingException ne) {
                throw new ConfigurationException("Failed to obtain queue session from pool", ne);
            }
    
            try {
                jmsDestination = (Destination) oJndiCtx.lookup(jmsDestinationName);
            }
            catch (NamingException nex) {
                try {
                    oJndiCtx = NamingContextPool.replaceNamingContext(oJndiCtx, environment);
                    jmsDestination = (Destination) oJndiCtx.lookup(jmsDestinationName);
                }
                catch (NamingException ne) {
                    if(JMSEpr.QUEUE_TYPE.equals(destType)) {
                        jmsDestination = jmsSession.createQueue(jmsDestinationName);
                    } else {
                        jmsDestination = jmsSession.createTopic(jmsDestinationName);
                    }
                }
            }
        } finally {
            NamingContextPool.releaseNamingContext(oJndiCtx) ;
        }

        if(durableSubscriptionName != null && jmsDestination instanceof Topic) {
        	jmsMessageConsumer = jmsSession.createDurableSubscriber((Topic)jmsDestination, durableSubscriptionName, _messageSelector, false);
        } else {
        	jmsMessageConsumer = jmsSession.createConsumer(jmsDestination, _messageSelector);
        }
    } // ________________________________

    /**
     * Receive one message and retry if connection
     *
     * @return javax.jms.Message - One input message, or null
     */
    protected javax.jms.Message receiveOne() {
        while (isRunning())
            try {
                javax.jms.Message ret = jmsMessageConsumer.receive(200);
                if (null != ret)
                    return ret;
            }
            catch (JMSException oJ) {
                if (_logger.isDebugEnabled()) {
                    _logger
                            .debug(
                                    "JMS error on receive.  Attempting JMS Destination reconnect.",
                                    oJ);
                }
                try {
                    prepareMessageReceiver();
                    errorDelay = 0;
                }
                // try to reconnect to the queue
                catch (Exception e) {
                    if (_logger.isDebugEnabled()) {
                        _logger.debug("Reconnecting to Queue", e);
                    }
                    if (errorDelay == 0) {
                        errorDelay = MIN_ERROR_DELAY;
                    } else if (errorDelay < MAX_ERROR_DELAY) {
                        errorDelay <<= 1;
                    }
                    _logger
                            .warn("Error reconnecting to Queue, backing off for "
                                    + errorDelay + " milliseconds");
                    waitForRunningStateChange(
                            ManagedLifecycleThreadState.STOPPING, errorDelay);
                }
            }
        return null;
    } // ________________________________

    protected final static Logger _logger = Logger
            .getLogger(JmsGatewayListener.class);

    protected String jmsDestinationName;

    protected JmsSession jmsSession;

    protected Destination jmsDestination;

    protected MessageConsumer jmsMessageConsumer;

    protected String _messageSelector;

    protected ConfigTree _config;

    protected String _serviceCategory, _serviceName;

    protected String _targetServiceCategory, _targetServiceName;

    protected EPR _myEpr;

    protected ServiceInvoker _serviceInvoker;
    
    protected String _composerName;

    protected Class _composerClass;

    protected Object _composer;

    protected Method _processMethod;

    protected JmsConnectionPool jmsConnectionPool;

    /**
     * The minimum error delay.
     */
    private static final long MIN_ERROR_DELAY = 1000;

    /**
     * The maximum error delay.
     */
    private static final long MAX_ERROR_DELAY = (MIN_ERROR_DELAY << 5);

    /**
     * The error delay.
     */
    private long errorDelay;

}
