/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.util.ClassUtil;
import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * A Helper classes used to validate xml files against supplied schemas.
 * 
 * @author $Revision$ $Id$
 */
@SuppressWarnings("unchecked")
public class XmlValidatorImpl implements XmlValidator {

	/**
	 * Our Logger
	 */
	private Logger log = Logger.getLogger(this.getClass());

	private String defaultSchema = "etc/schemas/xml/jbossesb-1.0.1.xsd";

	private boolean isValid = true;

	private Collection<String> validationResults = new HashSet();

	private Document xmlDocument;

	public XmlValidatorImpl() {
		File defaultSchemaFile = new File (defaultSchema);
		if (defaultSchemaFile.exists()) {
			log.debug("defaultSchema "+defaultSchema+" exists");
		}
	}

	public XmlValidatorImpl(InputSource xmlSource) {
		setXMLDocument(xmlSource);		

		File defaultSchemaFile = new File (defaultSchema);
		if (defaultSchemaFile.exists()) {
			log.debug("defaultSchema "+defaultSchema+" exists");
		}
	}

	/**
	 * Validation method used to validate an xml file against a default xsd
	 * (jbossesb-1.0.1.xsd).
	 * 
	 * @param xmlSource
	 *            The xml input source to be validated.
	 * @return Boolean true/false indicating successful validation.
	 * @throws XmlValidatorException
	 *             Failure during validation.
	 */
	public boolean validate(InputSource xmlSource) throws XmlValidatorException {
		StreamSource validationInputSource = null;

		InputStream validationInputStream = ClassUtil.getResourceAsStream(defaultSchema, getClass());
		// if this fails try using the
		if (validationInputStream==null) {
			File validationFile = new File(defaultSchema);
			log.debug("Validation file " + defaultSchema + " exists?:" + validationFile.exists());
			try {
				validationInputStream = new FileInputStream(validationFile);
			} catch (FileNotFoundException e) {
				log.error(e.getMessage(),e);
			}
		}
		if (validationInputStream==null) {
			log.warn("Could not obtain validation file " + defaultSchema);
		} else {
			log.debug("Reading validation info from " + defaultSchema);
			validationInputSource = new StreamSource(validationInputStream);
		}
		return validate(xmlSource,validationInputSource);
	}

	/**
	 * Validation method used to validate an xml input source against an xsd
	 * input source.
	 * 
	 * @param xmlSource
	 *            The xml input source to be validated.
	 * @param validationSource
	 *            The schema input source to validate against.
	 * @return Boolean true/false indicating successful validation.
	 * @throws XmlValidationException
	 *             Failure during validation.
	 */
	public boolean validate(InputSource xmlSource, StreamSource validationSource) throws XmlValidatorException 
	{
		try {
			Schema schema = null;
			SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
			if (validationSource == null) {
				Source schemaFile = new StreamSource(defaultSchema);
				try {
					schema = factory.newSchema(schemaFile);
				} catch (SAXException e) {
					log.error("Failed to validate xml: "+e);
					isValid = false;
				}
			}
			else {
				schema = factory.newSchema(validationSource);
			}
			DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
			builderFactory.setNamespaceAware(true);
			builderFactory.setSchema(schema);
			DocumentBuilder parser = null;
			try {
				parser = builderFactory.newDocumentBuilder();
				parser.setErrorHandler(new XmlErrorHandler());
			} catch (ParserConfigurationException e) {
				log.error("ParserConfigurationException occurred: " + e.getMessage(),e); 
				isValid = false;
			}
			try {
				xmlDocument = parser.parse(xmlSource); 
			} catch (IOException e) {
				log.error("parser.parse(): IOException occurred: " + e.getMessage(),e); 
				isValid = false;
			}      
		} catch (SAXException e) {
			log.error("SAXException occurred: " + e.getMessage());
			isValid = false;
		}
		return isValid;
	}

	class XmlErrorHandler implements ErrorHandler 
	{
		public void warning(SAXParseException exception) throws SAXException {
			// Bring things to a crashing halt
			validationResults.add("WARNING-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage());        
			log.warn("WARNING-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage());        
		}

		public void error(SAXParseException exception) throws SAXException {
			// Bring things to a crashing halt
			validationResults.add("ERROR-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage());        
			log.error(validationResults.size()+"-ERROR-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage());        
			isValid = false;
		}

		public void fatalError(SAXParseException exception) throws SAXException {
			// Bring things to a crashing halt
			validationResults.add("FATAL-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage());        
			log.fatal("FATAL-"+exception.getSystemId()+"-"+exception.getLineNumber()+":"+exception.getMessage());        
			isValid = false;
		}
	}

	public Collection<String> getValidationResults() 
	{
		return validationResults;
	}

	public void setXMLDocument(InputSource xmlSource) 
	{
		try {       
			DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder parser = builderFactory.newDocumentBuilder();

			xmlDocument = parser.parse(xmlSource); 
			log.info(xmlDocument.toString());
		} catch (SAXException e) {
			log.error(e.getMessage());
		} catch (ParserConfigurationException e) {
			log.error(e.getMessage());
		} catch (IOException e) {
			log.error(e.getMessage());
		}      
	}

	public Document getXMLDocument() {
		return xmlDocument;
	}

}
