/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.config;

import org.jboss.soa.esb.Service;

/**
 * Webservice information required to expose the service.
 *
 * @author <a href="mailto:Kevin.Conner@jboss.com">Kevin Conner</a>
 */
public class WebserviceInfo
{
    /**
     * The associated service details.
     */
    private final Service service ;
    /**
     * The input schema.
     */
    private final String inXsd ;
    /**
     * The output schema.
     */
    private final String outXsd ;
    /**
     * The fault schemas.
     */
    private final String faultXsd ;
    /**
     * The service description.
     */
    private final String description ;
    /**
     * The service request/response flag.
     */
    private final boolean requestResponse ;
    /**
     * The request location within the message.
     */
    private final String requestLocation ;
    /**
     * The response location within the message.
     */
    private final String responseLocation ;
    /**
     * Flag indicating the endpoint requires addressing.
     */
    private final boolean addressing ;

    /**
     * Construct the webservice information.
     * @param service The associated service details.
     * @param inXsd The input schema.
     * @param outXsd The output schema.
     * @param faultXsd The fault schemas.
     * @param description The service description.
     * @param requestResponse The service request/response flag.
     */
    public WebserviceInfo(final Service service, final String inXsd, final String outXsd,
        final String faultXsd, final String description, final boolean requestResponse)
    {
        this(service, inXsd, outXsd, faultXsd, description, requestResponse, null, null, false) ;
    }
    
    /**
     * Construct the webservice information.
     * @param service The associated service details.
     * @param inXsd The input schema.
     * @param outXsd The output schema.
     * @param faultXsd The fault schemas.
     * @param description The service description.
     * @param requestResponse The service request/response flag.
     * @param requestLocation The request location within the message.
     * @param responseLocation The response location within the message.
     * @param addressing Does the endpoint require addressing?
     */
    public WebserviceInfo(final Service service, final String inXsd, final String outXsd,
        final String faultXsd, final String description, final boolean requestResponse,
        final String requestLocation, final String responseLocation, final boolean addressing)
    {
        this.service = service ;
        this.inXsd = inXsd ;
        this.outXsd = outXsd ;
        this.faultXsd = faultXsd ;
        this.description = description ;
        this.requestResponse = requestResponse ;
        this.requestLocation = requestLocation ;
        this.responseLocation = responseLocation ;
        this.addressing = addressing ;
    }

    /**
     * Get the service details.
     * @return The service details.
     */
    public Service getService()
    {
        return service ;
    }

    /**
     * Get the input schema.
     * @return The input schema.
     */
    public String getInXsd()
    {
        return inXsd ;
    }

    /**
     * Get the output schema.
     * @return The output schema.
     */
    public String getOutXsd()
    {
        return outXsd ;
    }

    /**
     * Get the fault schema.
     * @return The fault schema.
     */
    public String getFaultXsd()
    {
        return faultXsd ;
    }

    /**
     * Get the service description.
     * @return The service description.
     */
    public String getDescription()
    {
        return description ;
    }

    /**
     * Is this a defined request/response service?
     * @return true if it is a request/response service, false otherwise.
     */
    public boolean isRequestResponse()
    {
        return requestResponse ;
    }

    /**
     * Get the request location within the message.
     * @return The request location.
     */
    public String getRequestLocation()
    {
        return requestLocation ;
    }

    /**
     * Get the response location within the message.
     * @return The response location.
     */
    public String getResponseLocation()
    {
        return responseLocation ;
    }

    /**
     * Does the endpoint require addressing?
     * @return
     */
    public boolean isAddressing()
    {
        return addressing ;
    }

    /**
     * Test for equality.
     * @param obj The object to compare with
     * @return true if equal, false otherwise.
     */
    public boolean equals(final Object obj)
    {
        if(obj instanceof WebserviceInfo)
        {
            final WebserviceInfo webserviceInfo = (WebserviceInfo)obj;
            return (service.equals(webserviceInfo.getService()) && objEquals(inXsd, webserviceInfo.inXsd) &&
                objEquals(outXsd, webserviceInfo.outXsd) && objEquals(faultXsd, webserviceInfo.faultXsd) &&
                objEquals(description, webserviceInfo.description) &&
                (requestResponse == webserviceInfo.isRequestResponse()) &&
                objEquals(requestLocation, webserviceInfo.requestLocation) &&
                objEquals(responseLocation, webserviceInfo.responseLocation) &&
                (addressing == webserviceInfo.isAddressing())) ;
        }

        return false ;
    }

    /**
     * Get the instance hash code.
     * @return the instance hash code.
     */
    public int hashCode()
    {
        return service.hashCode() ^ objHash(inXsd) ^ objHash(outXsd) ^ objHash(faultXsd) & objHash(description) ^ (requestResponse ? 0xdeafdeaf : 0) ^
            objHash(requestLocation) ^ objHash(responseLocation)  ^ (addressing ? 0xfeedfeed : 0);
    }

    /**
     * Get a string representation of this instance.
     * @return the string representation.
     */
    public String toString()
    {
        return service + objName("inXsd", inXsd) + objName("outXsd", outXsd) + objName("faultXsd" , faultXsd) + objName("description" , description) +
            (requestResponse ? ",requestResponse" : "") + objName("requestLocation", requestLocation) + objName("responseLocation", responseLocation) +
            (addressing ? ", addressing" : "") ;
    }

    /**
     * Generate a hashcode for an object.
     * @param value The object value or null.
     * @return The hashcode.
     */
    private int objHash(final String value)
    {
        return (value == null ? 0 : value.hashCode()) ;
    }

    /**
     * Generate a string representation for an object.
     * @param name The object name.
     * @param value The object value or null.
     * @return The string representation.
     */
    private String objName(final String name, final String value)
    {
        if (value != null)
        {
            return "," + name + "=" + value ;
        }
        else
        {
            return "" ;
        }
    }

    /**
     * Test for object equality.
     * @param lhs The object value or null.
     * @param rhs The object value or null.
     * @return true if equal, false otherwise.
     */
    private boolean objEquals(final Object lhs, final Object rhs)
    {
        if (lhs == null)
        {
            return (rhs == null) ;
        }
        else
        {
            return lhs.equals(rhs) ;
        }
    }
}
