package org.jboss.soa.esb.listeners.config;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.publish.ContractPublisher;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.publish.Publisher;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;

/**
 * Service contract publisher.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ServicePublisher {

    private String serviceName;
    private String category;
    private String description;
    private Publisher publisher;
    private static Map<ManagedLifecycleController, Map<Service, List<ServicePublisher>>> SERVICE_PUBLISHERS = new LinkedHashMap<ManagedLifecycleController, Map<Service, List<ServicePublisher>>>();
    private static ServicePublisherContractInfoCache CONTRACT_INFO_CACHE = new ServicePublisherContractInfoCache();

    /**
     * Public constructor.
     *
     * @param name     Service name.
     * @param category Service category.
     * @param publisher publisher implementation.
     */
    public ServicePublisher(String name, String category, Publisher publisher) {
        AssertArgument.isNotNullAndNotEmpty(name, "name");
        AssertArgument.isNotNullAndNotEmpty(category, "category");

        this.serviceName = name;
        this.category = category;
        this.publisher = publisher;
    }

    /**
     * Get the service description.
     * @return Service description.
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the service description.
     * @param description Service description.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Get the name of the Service.
     *
     * @return The service name.
     */
    public String getServiceName() {
        return serviceName;
    }

    /**
     * Get the Service category.
     *
     * @return The service category.
     */
    public String getCategory() {
        return category;
    }

    /**
     * Get the contract publisher for the Service associated with this publisher.
     *
     * @return The contract publisher.
     */
    public ContractPublisher getContractPublisher() {
        return (publisher instanceof ContractPublisher ? (ContractPublisher)publisher : null);
    }

    /**
     * Get the publisher for the Service associated with this publisher.
     *
     * @return The publisher.
     */
    public Publisher getPublisher() {
        return publisher ;
    }
    
    /**
     * Add the publisher into map for the specified service.
     * @param servicePublishers The service publishers
     * @param service The service name
     * @param publisher The publisher
     */
    private static void addPublisher(final Map<Service, List<ServicePublisher>> servicePublishers,
            final Service service, final ServicePublisher publisher)
    {
        final List<ServicePublisher> publishers = servicePublishers.get(service) ;
        if (publishers != null) {
            publishers.add(publisher);
        } else {
            final List<ServicePublisher> newPublishers = new ArrayList<ServicePublisher>() ;
            newPublishers.add(publisher) ;
            servicePublishers.put(service, newPublishers) ;
        }
    }

    /**
     * Add the service publishers to the controller map.
     * @param controller The controller for the services
     * @param servicePublishers The service publishers
     */
    public static synchronized void addServicePublishers(ManagedLifecycleController controller, Map<Service, List<ServicePublisher>> publishers) {
        final Map<Service, List<ServicePublisher>> currentMap = SERVICE_PUBLISHERS.get(controller) ;
        if (currentMap != null) {
            for(Map.Entry<Service, List<ServicePublisher>> entry: publishers.entrySet()) {
                final Service service = entry.getKey() ;
                final List<ServicePublisher> currentList = currentMap.get(service) ;
                if (currentList != null) {
                    currentList.addAll(entry.getValue()) ;
                } else {
                    currentMap.put(service, entry.getValue()) ;
                }
            }
        } else {
            SERVICE_PUBLISHERS.put(controller, publishers) ;
        }
    }

    /**
     * Get the full list of publishers registered against all active {@link ManagedLifecycleController}
     * instances.
     * @return The full list of publishers.
     */
    public static synchronized List<ServicePublisher> getServicePublishers() {
        List<ServicePublisher> publishers = new ArrayList<ServicePublisher>();
        Collection<Map<Service, List<ServicePublisher>>> allPublishers = SERVICE_PUBLISHERS.values();

        for(Map<Service, List<ServicePublisher>> servicePublishers: allPublishers) {
            for(List<ServicePublisher> curPublisherList : servicePublishers.values()) {
                publishers.addAll(curPublisherList);
            }
        }

        return publishers;
    }

    /**
     * Remove the service publications for the services under the control of the supplied controller.
     * @param controller Controller.
     */
    public static synchronized void removeServicePublishers(ManagedLifecycleController controller) {
    	final Map<Service, List<ServicePublisher>> removedMap = SERVICE_PUBLISHERS.remove(controller);
    	if (removedMap != null) {
    		synchronized (CONTRACT_INFO_CACHE) {
	    		for (List<ServicePublisher> servicePublisherList : removedMap.values()) {
	    			if (servicePublisherList != null) {
	    				for (ServicePublisher servicePublisher : servicePublisherList) {
	    					if (servicePublisher != null) {
	    						CONTRACT_INFO_CACHE.removeContractInfos(servicePublisher);
	    					}
	    				}
	    			}
	    		}
    		}
    	}
    }

    /**
     * Add external contract reference publishers to the list of publishers
     * @param controller The controller
     * @param publishers
     */
    public static synchronized void addContractReferencePublishers(final ManagedLifecycleController controller, final List<? extends ContractReferencePublisher> publishers) {
        if (publishers != null) {
            final Map<Service, List<ServicePublisher>> servicePublishers = new LinkedHashMap<Service, List<ServicePublisher>>() ;
            for(ContractReferencePublisher publisher: publishers) {
                final Service service = publisher.getService() ;
                final ServicePublisher servicePublisher = new ServicePublisher(service.getName(), service.getCategory(), publisher) ;
                servicePublisher.setDescription(publisher.getDescription()) ;
                addPublisher(servicePublishers, service, servicePublisher) ;
            }
            addServicePublishers(controller, servicePublishers) ;
        }
    }

    /**
     * Get the list of registered services
     * @return The list of services.
     */
    public static synchronized List<Service> getServices() {
        LinkedHashSet<Service> services = new LinkedHashSet<Service>();
        Collection<Map<Service, List<ServicePublisher>>> allPublishers = SERVICE_PUBLISHERS.values();

        for(Map<Service, List<ServicePublisher>> servicePublishers: allPublishers) {
            services.addAll(servicePublishers.keySet()) ;
        }

        return new ArrayList<Service>(services);
    }

    /**
     * Get the list of publishers registered against the specified {@link Service}
     * @return The list of publishers.
     */
    public static synchronized List<ServicePublisher> getServicePublishers(final Service service) {
        List<ServicePublisher> publishers = new ArrayList<ServicePublisher>();
        Collection<Map<Service, List<ServicePublisher>>> allPublishers = SERVICE_PUBLISHERS.values();

        for(Map<Service, List<ServicePublisher>> servicePublishers: allPublishers) {
            final List<ServicePublisher> curPublisherList = servicePublishers.get(service) ;
            if (curPublisherList != null) {
                publishers.addAll(curPublisherList);
            }
        }

        return publishers;
    }
    
    // used by contract.jsp
    public static ServicePublisherContractInfoCache getContractInfoCache() {
    	return CONTRACT_INFO_CACHE;
    }
}
