/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.config.model.ModelException;
import org.jboss.soa.esb.listeners.config.model.ModelParser;
import org.w3c.dom.Document;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;

/**
 * ESB Configuration Generator.
 * <p/>
 * Generates a legacy/deprecated configuration fileset in accordance with the
 * {@link org.jboss.soa.esb.helpers.ConfigTree} approach to ESB configuration.  The
 * generator input should be a configuration that has been validated against the
 * JBoss ESB configuration schema (XMLSchema).
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class Generator {
	
	private static Logger logger = Logger.getLogger(Generator.class);
	
	/**
	 * Gateway Listeners config file.
	 */
	public static final String ESB_CONFIG_GATEWAY_XML_FILE = "jbossesb-gateway.xml";
	/**
	 * ESB Aware Listeners config file.
	 */
	public static final String ESB_CONFIG_XML_FILE = "jbossesb-listener.xml";
	/**
	 * Configuration model.
	 */
	private ModelAdapter model;

   private OutputStream gatewayOutputStream;

   private OutputStream listenerOutputStream;

   public Generator(InputStream config, OutputStream listenerOutputStream, OutputStream gatewayOutputStream) throws ConfigurationException, IOException
   {
      this(config);
      this.gatewayOutputStream = gatewayOutputStream;
      this.listenerOutputStream = listenerOutputStream;
   }

   /**
	 * Public constructor.
	 * @param config The input configuration stream.
	 * @throws ConfigurationException Bad listener ESB configuration. 
	 * @throws IOException Unable to read the ESB listener configuration.
	 */
	public Generator(InputStream config) throws ConfigurationException, IOException {
		if(config == null) {
			throw new IllegalArgumentException("null 'config' arg in constructor call.");
		}
		
		// Parse the config into our internal model instance...
       model = parseConfig(config);
       model.verifyScheduleProviderConfig();
   }

    /**
    * Parse the supplied config into a config model.
    * @param config The input configuration stream.
    * @return Config model.
    * @throws ConfigurationException Bad listener ESB configuration.
    * @throws IOException Unable to read the ESB listener configuration.
    */
   public static ModelAdapter parseConfig(InputStream config) throws IOException, ConfigurationException {
       try {
           return ModelParser.getParser().parse(new InputStreamReader(config)) ;
       } catch (final ModelException me) {
           throw new ConfigurationException("Error while processing ESB Listener configuration stream.", me);
       }
   }

   /**
    * Generate the configuration set in the supplied output directory and store it in the member outputstreams.
    * @throws ConfigurationException Failed to generate configuration set.
    */
   public void generate() throws ConfigurationException {
      Document awareConfig = model.generateESBAwareConfig();

      YADOMUtil.serialize(awareConfig, listenerOutputStream);

      Document gatewayConfig = model.generateGatewayConfig();

      YADOMUtil.serialize(gatewayConfig, gatewayOutputStream);
   }

	/**
	 * Generate the configuration set in the supplied output directory.
	 * <p/>
	 * The names of the generated files will be esb-config.xml (ESB Aware Listeners configuration)
	 * and esb-config-gateway.xml (Gateway Listeners configuration).
	 * 
	 * @param outdir The output directory to where the configuration set is to be generated.
	 * The directory will be created if it doesn't already exist.
	 * @throws ConfigurationException Failed to generate configuration set.
	 */
	public void generate(File outdir) throws ConfigurationException {
		if(outdir == null) {
			throw new IllegalArgumentException("null 'outdir' arg in method call.");
		}
		if(!outdir.exists()) {
			outdir.mkdirs();
			if(!outdir.exists()) {
				throw new IllegalArgumentException("Failed to create 'outdir' [" + outdir.getAbsolutePath() + "].");
			}
		}
		if(!outdir.isDirectory()) {
			throw new IllegalArgumentException("'outdir' exists, but is not a directory.");
		}

		Document awareConfig = model.generateESBAwareConfig();

		logger.info("Serializing ESB Aware Listener 'ConfigTree' configuration to [" + (new File(outdir, ESB_CONFIG_XML_FILE)).getAbsolutePath() + "].");
		YADOMUtil.serialize(awareConfig, outdir, ESB_CONFIG_XML_FILE);

		Document gatewayConfig = model.generateGatewayConfig();
		
		logger.info("Serializing ESB Gateway 'ConfigTree' configuration to [" + (new File(outdir, ESB_CONFIG_GATEWAY_XML_FILE)).getAbsolutePath() + "].");
		YADOMUtil.serialize(gatewayConfig, outdir, ESB_CONFIG_GATEWAY_XML_FILE);
	}

	/**
	 * Get the model instance associate with the configuration used to create this Generator instance. 
	 * @return Returns the model.
	 */
	public ModelAdapter getModel() {
		return model;
	}
}
